"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dataSourceMachine = exports.createDataSourceMachineImplementations = void 0;
var _xstate = require("xstate5");
var _xstateUtils = require("@kbn/xstate-utils");
var _lodash = require("lodash");
var _eui = require("@elastic/eui");
var _data_collector_actor = require("./data_collector_actor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const dataSourceColors = (0, _eui.euiPaletteColorBlindBehindText)();
let colorCounter = 0;

// Retain a global state for assigning unique colors to data sources
function getNewColor() {
  const color = dataSourceColors[colorCounter % dataSourceColors.length];
  colorCounter += 1;
  return color;
}
const dataSourceMachine = exports.dataSourceMachine = (0, _xstate.setup)({
  types: {
    input: {},
    context: {},
    events: {}
  },
  actors: {
    collectData: (0, _xstateUtils.getPlaceholderFor)(_data_collector_actor.createDataCollectorActor)
  },
  delays: {
    dataSourceChangeDebounceTime: 800
  },
  actions: {
    notifyDataCollectionFailure: (0, _xstateUtils.getPlaceholderFor)(_data_collector_actor.createDataCollectionFailureNofitier),
    storeDataSource: (0, _xstate.assign)(({
      context
    }, params) => ({
      dataSource: {
        ...params.dataSource,
        id: context.dataSource.id
      }
    })),
    storeData: (0, _xstate.assign)((_, params) => ({
      data: params.data
    })),
    toggleDataSourceActivity: (0, _xstate.assign)(({
      context
    }) => ({
      dataSource: {
        ...context.dataSource,
        enabled: !context.dataSource.enabled
      }
    })),
    notifyParent: (0, _xstate.sendTo)(({
      context
    }) => context.parentRef, ({
      context
    }, params) => ({
      type: params.eventType,
      id: context.dataSource.id
    }))
  },
  guards: {
    isEnabled: ({
      context
    }) => context.dataSource.enabled,
    isDeletable: ({
      context
    }) => context.dataSource.type !== 'random-samples',
    // We don't allow deleting the random-sample source to always have a data source available
    isValidData: (_, params) => Array.isArray(params.data),
    shouldCollectData: ({
      context,
      event
    }) => {
      (0, _xstate.assertEvent)(event, 'dataSource.change');
      /**
       * Determines if the dataSource update contains substantive changes.
       * Ignores cosmetic changes like name updates that don't affect functionality.
       */
      const ignoredProps = ['name'];
      return !(0, _lodash.isEqual)((0, _lodash.omit)(context.dataSource, ignoredProps), (0, _lodash.omit)(event.dataSource, ignoredProps));
    }
  }
}).createMachine({
  /** @xstate-layout N4IgpgJg5mDOIC5QQIYBcUGUD2BXATgMZgDEqGOBxAdBGADZhpgDaADALqKgAO2sASzQDsAO24gAHogCcMgMzUZAdlUAmNcpkBWACwBGfQBoQAT0S6AbIrYAOW5pn7ry2-rbKAvp5PkseIlI-SkDqNGwoKEYAQUJhADchU3YuJBA+QWExCWkENQNqZQ8ZNQUithldE3MEXTVLamttfRk3S202Uo9vX3R-KiC+kJpwyJi4gUS0ZP1U3n4hEXE03OtbajV5W11leTr9eWU1aot6xvlm1ucO0p6QYICaOmZ8AFsBUQ+oEhSJDMXsitEABafRqahsfS2eRsSz6ZTaaEIzYnBCI5TUeQySH6XQXEpwrw+e5DR5gWhMMBvD5fH6zP4LLLLUC5ZzrXH2WwySw6KHaZRVMyIRENLRqbSVWGWIpqO4PAbUMCiFAAI0YEDIpIV+DAADMdbAABa-NL-Jk5RDuNjUDm6Wwi3YVfTaVFIiH83TNTryUpqfRyrWhJWq9Waihk6iEQ0oUQwE3zTJLC1ojqFZS49MeXYdF1ChAlfQ2uoVG5aOQyAPhhXBtWQMP9UJRmNx+mmxlJoEIK3aG2tBE7O0HPGogtFtQlzq2SsNmg19UUlV4USEL4AYWjsbgJEksAwzGoKF1LwAFPLAuvm2AACJgRe4ZdgAAqAleYAAlPXhuS55AF0uV7GF6brA8bpO2gIsog8jQRCVh4jC456LYyiWKi7gOEo8jwkhKjuK405foqyq1hA1D0NgKAQF8V59CQVL4Ng+DUDw9DoLqjGvLQfSrtg9CMHEjGxOE+CgWaHaQQg0I2to1gIZ0zjSsoaFsDBbCdCplwwgc3jEqI2B0PAaRnsQDKJhBUggnC1C2Gp0qWDsOibChqLQTINoXOOyFFPClgERGzxUu8nyxqZALMhZCCgjY8KuDJGZOChxx5p6hbtNoFzpR0dTjrofnVsR6qheanZRdZtkoQ52hORKqIytQ2genCMKqPy2h5UGBW-gIECMEV4kRc6GJYh44oOLsliaGhGjgnYpTws4nTtESvRVh1Ia-nQd7LmuG4wIZCZhcmE1uc6dQ7DJRSejIykCpiDhsJ6dq6E41jtbOnWkeRlHUX0fXmbkey9gomxVQ9iWCjUJTWnizSGItMnbG95JUbA60QH94UA9YGyGPaOzOs6kIQ8K9pFkTHQoQ42nEsZyMMJS6NtmZmOWip6ytRcWJ2viqF5jJ1rDWCmhsIilQVjpQA */
  id: 'dataSource',
  context: ({
    input
  }) => ({
    parentRef: input.parentRef,
    dataSource: input.dataSource,
    streamName: input.streamName,
    uiAttributes: {
      color: getNewColor()
    },
    data: []
  }),
  initial: 'determining',
  on: {
    'dataSource.delete': {
      guard: 'isDeletable',
      target: '.deleted'
    },
    'dataSource.toggleActivity': [{
      guard: 'isEnabled',
      target: '.disabled',
      actions: [{
        type: 'toggleDataSourceActivity'
      }, {
        type: 'notifyParent',
        params: {
          eventType: 'dataSource.change'
        }
      }]
    }, {
      target: '.enabled',
      actions: [{
        type: 'toggleDataSourceActivity'
      }, {
        type: 'notifyParent',
        params: {
          eventType: 'dataSource.change'
        }
      }]
    }]
  },
  states: {
    determining: {
      always: [{
        target: 'enabled',
        guard: 'isEnabled'
      }, {
        target: 'disabled'
      }]
    },
    enabled: {
      initial: 'loadingData',
      on: {
        'dataSource.refresh': '.loadingData',
        'dataSource.change': [{
          guard: 'shouldCollectData',
          target: '.debouncingChanges',
          reenter: true,
          actions: [{
            type: 'storeDataSource',
            params: ({
              event
            }) => event
          }, {
            type: 'notifyParent',
            params: {
              eventType: 'dataSource.change'
            }
          }]
        }, {
          actions: [{
            type: 'storeDataSource',
            params: ({
              event
            }) => event
          }, {
            type: 'notifyParent',
            params: {
              eventType: 'dataSource.change'
            }
          }]
        }]
      },
      exit: [{
        type: 'notifyParent',
        params: {
          eventType: 'dataSource.dataChange'
        }
      }],
      states: {
        idle: {},
        debouncingChanges: {
          after: {
            dataSourceChangeDebounceTime: 'loadingData'
          }
        },
        loadingData: {
          invoke: {
            id: 'dataCollectorActor',
            src: 'collectData',
            input: ({
              context
            }) => ({
              dataSource: context.dataSource,
              streamName: context.streamName
            }),
            onSnapshot: {
              guard: {
                type: 'isValidData',
                params: ({
                  event
                }) => ({
                  data: event.snapshot.context
                })
              },
              target: 'idle',
              actions: [{
                type: 'storeData',
                params: ({
                  event
                }) => {
                  var _event$snapshot$conte;
                  return {
                    data: (_event$snapshot$conte = event.snapshot.context) !== null && _event$snapshot$conte !== void 0 ? _event$snapshot$conte : []
                  };
                }
              }, {
                type: 'notifyParent',
                params: {
                  eventType: 'dataSource.dataChange'
                }
              }]
            },
            onError: {
              target: 'idle',
              actions: [{
                type: 'storeData',
                params: () => ({
                  data: []
                })
              }, {
                type: 'notifyParent',
                params: {
                  eventType: 'dataSource.dataChange'
                }
              }, {
                type: 'notifyDataCollectionFailure'
              }]
            }
          }
        }
      }
    },
    disabled: {},
    deleted: {
      id: 'deleted',
      type: 'final',
      entry: [{
        type: 'notifyParent',
        params: {
          eventType: 'dataSource.delete'
        }
      }]
    }
  }
});
const createDataSourceMachineImplementations = ({
  data,
  toasts
}) => ({
  actors: {
    collectData: (0, _data_collector_actor.createDataCollectorActor)({
      data
    })
  },
  actions: {
    notifyDataCollectionFailure: (0, _data_collector_actor.createDataCollectionFailureNofitier)({
      toasts
    })
  }
});
exports.createDataSourceMachineImplementations = createDataSourceMachineImplementations;