"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initUiSettings = exports.getDefaultValueReportSettings = exports.getDefaultAIConnectorSetting = void 0;
var _i18n = require("@kbn/i18n");
var _configSchema = require("@kbn/config-schema");
var _constants = require("../common/constants");
var _rule_monitoring = require("../common/api/detection_engine/rule_monitoring");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This helper is used to preserve settings order in the UI
 *
 * @param settings - UI settings config
 * @returns Settings config with the order field added
 */
const orderSettings = settings => {
  return Object.fromEntries(Object.entries(settings).map(([id, setting], index) => [id, {
    ...setting,
    order: index
  }]));
};
const initUiSettings = (uiSettings, experimentalFeatures, validationsEnabled) => {
  const securityUiSettings = {
    [_constants.DEFAULT_APP_REFRESH_INTERVAL]: {
      type: 'json',
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultRefreshIntervalLabel', {
        defaultMessage: 'Time filter refresh interval'
      }),
      value: `{
  "pause": ${_constants.DEFAULT_INTERVAL_PAUSE},
  "value": ${_constants.DEFAULT_INTERVAL_VALUE}
}`,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultRefreshIntervalDescription', {
        defaultMessage: '<p>Default refresh interval for the Security time filter, in milliseconds.</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.object({
        value: _configSchema.schema.number(),
        pause: _configSchema.schema.boolean()
      }),
      solutionViews: ['classic', 'security']
    },
    [_constants.DEFAULT_APP_TIME_RANGE]: {
      type: 'json',
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultTimeRangeLabel', {
        defaultMessage: 'Time filter period'
      }),
      value: `{
  "from": "${_constants.DEFAULT_FROM}",
  "to": "${_constants.DEFAULT_TO}"
}`,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultTimeRangeDescription', {
        defaultMessage: '<p>Default period of time in the Security time filter.</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.object({
        from: _configSchema.schema.string(),
        to: _configSchema.schema.string()
      }),
      solutionViews: ['classic', 'security']
    },
    [_constants.DEFAULT_INDEX_KEY]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultIndexLabel', {
        defaultMessage: 'Elasticsearch indices'
      }),
      sensitive: true,
      value: _constants.DEFAULT_INDEX_PATTERN,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultIndexDescription', {
        defaultMessage: '<p>Comma-delimited list of Elasticsearch indices from which the Security app collects events.</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: validationsEnabled ? _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        maxSize: 50
      }) : _configSchema.schema.arrayOf(_configSchema.schema.string()),
      solutionViews: ['classic', 'security']
    },
    [_constants.DEFAULT_THREAT_INDEX_KEY]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultThreatIndexLabel', {
        defaultMessage: 'Threat indices'
      }),
      sensitive: true,
      value: _constants.DEFAULT_THREAT_INDEX_VALUE,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultThreatIndexDescription', {
        defaultMessage: '<p>Comma-delimited list of Threat Intelligence indices from which the Security app collects indicators.</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: validationsEnabled ? _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        maxSize: 10
      }) : _configSchema.schema.arrayOf(_configSchema.schema.string()),
      solutionViews: ['classic', 'security']
    },
    [_constants.DEFAULT_ANOMALY_SCORE]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultAnomalyScoreLabel', {
        defaultMessage: 'Anomaly threshold'
      }),
      value: 50,
      type: 'number',
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultAnomalyScoreDescription', {
        defaultMessage: '<p>Value above which Machine Learning job anomalies are displayed in the Security app.</p><p>Valid values: 0 to 100.</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: validationsEnabled ? _configSchema.schema.number({
        max: 100,
        min: 0
      }) : _configSchema.schema.number(),
      solutionViews: ['classic', 'security']
    },
    [_constants.ENABLE_NEWS_FEED_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableNewsFeedLabel', {
        defaultMessage: 'News feed'
      }),
      value: true,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableNewsFeedDescription', {
        defaultMessage: '<p>Enables the News feed</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      type: 'boolean',
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.boolean(),
      solutionViews: ['classic', 'security']
    },
    [_constants.EXCLUDE_COLD_AND_FROZEN_TIERS_IN_ANALYZER]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.excludeColdAndFrozenTiersInAnalyzer', {
        defaultMessage: 'Exclude cold and frozen tiers in Analyzer'
      }),
      value: false,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.excludeColdAndFrozenTiersInAnalyzerDescription', {
        defaultMessage: '<p>When enabled, cold and frozen tiers will be skipped in analyzer queries</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      type: 'boolean',
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.boolean(),
      solutionViews: ['classic', 'security']
    },
    [_constants.ENABLE_GRAPH_VISUALIZATION_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableGraphVisualizationLabel', {
        defaultMessage: 'Enable graph visualization'
      }),
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableGraphVisualizationDescription', {
        defaultMessage: `Enable the Graph Visualization feature within the Security Solution.`
      }),
      type: 'boolean',
      value: false,
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.boolean(),
      solutionViews: ['classic', 'security'],
      technicalPreview: true
    },
    [_constants.ENABLE_ASSET_INVENTORY_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableAssetInventoryLabel', {
        defaultMessage: 'Enable Security Asset Inventory'
      }),
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableAssetInventoryDescription', {
        defaultMessage: `Enable the Asset Inventory experience within the Security Solution. When enabled, you can access the new Inventory feature through the Security Solution navigation. Note: Disabling this setting will not disable the Entity Store or clear persistent Entity metadata. To manage or disable the Entity Store, please visit the Entity Store Management page.`
      }),
      type: 'boolean',
      value: false,
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.boolean(),
      solutionViews: ['classic', 'security'],
      technicalPreview: true
    },
    ...(experimentalFeatures.siemReadinessDashboard && {
      [_constants.ENABLE_SIEM_READINESS_SETTING]: {
        name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableSiemReadinessLabel', {
          defaultMessage: 'Enable SIEM Readiness Dashboard'
        }),
        description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableSiemReadinessDescription', {
          defaultMessage: `Enable the SIEM Readiness Dashboard within Security Solution. When enabled, you can access the new SIEM Readiness page through the navigation menu.`
        }),
        type: 'boolean',
        value: false,
        category: [_constants.APP_ID],
        requiresPageReload: true,
        schema: _configSchema.schema.boolean(),
        solutionViews: ['classic', 'security'],
        technicalPreview: true
      }
    }),
    [_constants.ENABLE_CLOUD_CONNECTOR_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableAssetInventoryLabel', {
        defaultMessage: 'Enable Cloud Connector'
      }),
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableAssetInventoryDescription', {
        defaultMessage: `Enable the Cloud Connector experience within the Security Solution. When enabled, you can access the new Cloud Connector feature through the setting up an Agentless CSPM or Asset Inventory Integration.`
      }),
      type: 'boolean',
      value: true,
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.boolean(),
      solutionViews: ['classic', 'security'],
      technicalPreview: true
    },
    [_constants.DEFAULT_RULES_TABLE_REFRESH_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.rulesTableRefresh', {
        defaultMessage: 'Rules auto refresh'
      }),
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.rulesTableRefreshDescription', {
        defaultMessage: '<p>Enables auto refresh on the rules and monitoring tables, in milliseconds</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      type: 'json',
      value: `{
  "on": ${_constants.DEFAULT_RULE_REFRESH_INTERVAL_ON},
  "value": ${_constants.DEFAULT_RULE_REFRESH_INTERVAL_VALUE}
}`,
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.object({
        value: _configSchema.schema.number({
          min: 60000
        }),
        on: _configSchema.schema.boolean()
      }),
      solutionViews: ['classic', 'security']
    },
    [_constants.NEWS_FEED_URL_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.newsFeedUrl', {
        defaultMessage: 'News feed URL'
      }),
      value: _constants.NEWS_FEED_URL_SETTING_DEFAULT,
      sensitive: true,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.newsFeedUrlDescription', {
        defaultMessage: '<p>News feed content will be retrieved from this URL</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.string(),
      solutionViews: ['classic', 'security']
    },
    [_constants.IP_REPUTATION_LINKS_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.ipReputationLinks', {
        defaultMessage: 'IP Reputation Links'
      }),
      value: _constants.IP_REPUTATION_LINKS_SETTING_DEFAULT,
      type: 'json',
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.ipReputationLinksDescription', {
        defaultMessage: 'Array of URL templates to build the list of reputation URLs to be displayed on the IP Details page.'
      }),
      sensitive: true,
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.arrayOf(_configSchema.schema.object({
        name: _configSchema.schema.string(),
        url_template: _configSchema.schema.string()
      })),
      solutionViews: ['classic', 'security']
    },
    [_constants.ENABLE_CCS_READ_WARNING_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableCcsReadWarningLabel', {
        defaultMessage: 'CCS Rule Privileges Warning'
      }),
      value: true,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableCcsWarningDescription', {
        defaultMessage: '<p>Enables privilege check warnings in rules for CCS indices</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      type: 'boolean',
      category: [_constants.APP_ID],
      requiresPageReload: false,
      schema: _configSchema.schema.boolean(),
      solutionViews: ['classic', 'security']
    },
    [_constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.suppressionBehaviorOnAlertClosureLabel', {
        defaultMessage: 'Default suppression behavior on alert closure'
      }),
      value: _constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING_ENUM.RestartWindow,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.suppressionBehaviorOnAlertClosureDescription', {
        defaultMessage: 'If an alert is closed while suppression is active, you can choose whether suppression continues or resets.'
      }),
      type: 'select',
      schema: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING_ENUM.RestartWindow), _configSchema.schema.literal(_constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING_ENUM.ContinueWindow)]),
      options: [_constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING_ENUM.RestartWindow, _constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING_ENUM.ContinueWindow],
      optionLabels: {
        [_constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING_ENUM.RestartWindow]: _i18n.i18n.translate('xpack.securitySolution.uiSettings.suppressionBehaviorOnAlertClosure.restart', {
          defaultMessage: 'Restart suppression'
        }),
        [_constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING_ENUM.ContinueWindow]: _i18n.i18n.translate('xpack.securitySolution.uiSettings.suppressionBehaviorOnAlertClosure.continue', {
          defaultMessage: 'Continue suppression until window ends'
        })
      },
      category: [_constants.APP_ID],
      requiresPageReload: false
    },
    [_constants.SHOW_RELATED_INTEGRATIONS_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.showRelatedIntegrationsLabel', {
        defaultMessage: 'Related integrations'
      }),
      value: true,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.showRelatedIntegrationsDescription', {
        defaultMessage: '<p>Shows related integrations on the rules and monitoring tables</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      type: 'boolean',
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.boolean(),
      solutionViews: ['classic', 'security']
    },
    [_constants.DEFAULT_ALERT_TAGS_KEY]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultAlertTagsLabel', {
        defaultMessage: 'Alert tagging options'
      }),
      sensitive: true,
      value: _constants.DEFAULT_ALERT_TAGS_VALUE,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultAlertTagsDescription', {
        defaultMessage: '<p>List of tag options for use with alerts generated by Security Solution rules.</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.arrayOf(_configSchema.schema.string()),
      solutionViews: ['classic', 'security']
    },
    [_constants.EXCLUDED_DATA_TIERS_FOR_RULE_EXECUTION]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.excludedDataTiersForRuleExecutionLabel', {
        defaultMessage: 'Exclude cold or frozen data tier from rule execution'
      }),
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.excludedDataTiersForRuleExecutionDescription', {
        defaultMessage: `
          When configured, events from the specified data tiers are not searched during rules executions.
          <br/>This might help to improve rule performance or reduce execution time.
          <br/>If you specify multiple data tiers, separate values with commas. For example: data_frozen,data_cold`
      }),
      type: 'array',
      schema: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.literal('data_cold'), _configSchema.schema.literal('data_frozen')])),
      value: [],
      category: [_constants.APP_ID],
      requiresPageReload: false,
      solutionViews: ['classic', 'security']
    },
    [_constants.ENABLE_PRIVILEGED_USER_MONITORING_SETTING]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enablePrivilegedUserMonitoringLabel', {
        defaultMessage: 'Privileged user monitoring'
      }),
      value: true,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enablePrivilegedUserMonitoringDescription', {
        defaultMessage: '<p>Enables the privileged user monitoring dashboard and onboarding experience which are in technical preview.</p>',
        values: {
          p: chunks => `<p>${chunks}</p>`
        }
      }),
      type: 'boolean',
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.boolean(),
      solutionViews: ['classic', 'security']
    },
    ...(experimentalFeatures.disableESQLRiskScoring ? {} : {
      [_constants.ENABLE_ESQL_RISK_SCORING]: {
        name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableEsqlRiskScoringLabel', {
          defaultMessage: 'Enable ESQL-based risk scoring'
        }),
        value: true,
        description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.enableEsqlRiskScoringDescription', {
          defaultMessage: '<p>Enables risk scoring based on ESQL queries. Disabling this will revert to using scripted metrics</p>',
          values: {
            p: chunks => `<p>${chunks}</p>`
          }
        }),
        type: 'boolean',
        category: [_constants.APP_ID],
        requiresPageReload: true,
        schema: _configSchema.schema.boolean(),
        solutionViews: ['classic', 'security']
      }
    }),
    ...(experimentalFeatures.extendedRuleExecutionLoggingEnabled ? {
      [_constants.EXTENDED_RULE_EXECUTION_LOGGING_ENABLED_SETTING]: {
        name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingEnabledLabel', {
          defaultMessage: 'Extended rule execution logging'
        }),
        description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingEnabledDescription', {
          defaultMessage: '<p>Enables extended rule execution logging to .kibana-event-log-* indices. Shows plain execution events on the Rule Details page.</p>',
          values: {
            p: chunks => `<p>${chunks}</p>`
          }
        }),
        type: 'boolean',
        schema: _configSchema.schema.boolean(),
        value: true,
        category: [_constants.APP_ID],
        requiresPageReload: false,
        solutionViews: ['classic', 'security']
      },
      [_constants.EXTENDED_RULE_EXECUTION_LOGGING_MIN_LEVEL_SETTING]: {
        name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingMinLevelLabel', {
          defaultMessage: 'Extended rule execution logging: min level'
        }),
        description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingMinLevelDescription', {
          defaultMessage: '<p>Sets minimum log level starting from which rules will write extended logs to .kibana-event-log-* indices. This affects only events of type Message, other events are being written to .kibana-event-log-* regardless of this setting and their log level.</p>',
          values: {
            p: chunks => `<p>${chunks}</p>`
          }
        }),
        type: 'select',
        schema: _configSchema.schema.oneOf([_configSchema.schema.literal(_rule_monitoring.LogLevelSetting.off), _configSchema.schema.literal(_rule_monitoring.LogLevelSetting.error), _configSchema.schema.literal(_rule_monitoring.LogLevelSetting.warn), _configSchema.schema.literal(_rule_monitoring.LogLevelSetting.info), _configSchema.schema.literal(_rule_monitoring.LogLevelSetting.debug), _configSchema.schema.literal(_rule_monitoring.LogLevelSetting.trace)]),
        value: _rule_monitoring.LogLevelSetting.error,
        options: [_rule_monitoring.LogLevelSetting.off, _rule_monitoring.LogLevelSetting.error, _rule_monitoring.LogLevelSetting.warn, _rule_monitoring.LogLevelSetting.info, _rule_monitoring.LogLevelSetting.debug, _rule_monitoring.LogLevelSetting.trace],
        optionLabels: {
          [_rule_monitoring.LogLevelSetting.off]: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingMinLevelOff', {
            defaultMessage: 'Off'
          }),
          [_rule_monitoring.LogLevelSetting.error]: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingMinLevelError', {
            defaultMessage: 'Error'
          }),
          [_rule_monitoring.LogLevelSetting.warn]: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingMinLevelWarn', {
            defaultMessage: 'Warn'
          }),
          [_rule_monitoring.LogLevelSetting.info]: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingMinLevelInfo', {
            defaultMessage: 'Info'
          }),
          [_rule_monitoring.LogLevelSetting.debug]: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingMinLevelDebug', {
            defaultMessage: 'Debug'
          }),
          [_rule_monitoring.LogLevelSetting.trace]: _i18n.i18n.translate('xpack.securitySolution.uiSettings.extendedRuleExecutionLoggingMinLevelTrace', {
            defaultMessage: 'Trace'
          })
        },
        category: [_constants.APP_ID],
        requiresPageReload: false,
        solutionViews: ['classic', 'security']
      }
    } : {})
  };
  uiSettings.register(orderSettings(securityUiSettings));
};
exports.initUiSettings = initUiSettings;
const getDefaultAIConnectorSetting = (connectors, readonlyMode) => {
  var _connectors$at;
  return {
    [_constants.DEFAULT_AI_CONNECTOR]: {
      name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultAIConnectorLabel', {
        defaultMessage: 'Default AI Connector'
      }),
      // TODO, make Elastic LLM the default value once fully available in serverless
      value: (_connectors$at = connectors.at(0)) === null || _connectors$at === void 0 ? void 0 : _connectors$at.id,
      description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultAIConnectorDescription', {
        defaultMessage: 'Default AI connector for serverless AI features (Elastic AI SOC Engine)'
      }),
      type: 'select',
      options: connectors.map(({
        id
      }) => id),
      optionLabels: Object.fromEntries(connectors.map(({
        id,
        name
      }) => [id, name])),
      category: [_constants.APP_ID],
      requiresPageReload: true,
      schema: _configSchema.schema.string(),
      solutionViews: ['classic', 'security'],
      readonlyMode,
      readonly: readonlyMode !== undefined
    }
  };
};
exports.getDefaultAIConnectorSetting = getDefaultAIConnectorSetting;
const getDefaultValueReportSettings = () => ({
  [_constants.DEFAULT_VALUE_REPORT_MINUTES]: {
    name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultValueMinutesLabel', {
      defaultMessage: 'Value report minutes per alert'
    }),
    value: 8,
    description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultValueMinutesDescription', {
      defaultMessage: 'The average review time for an analyst to review an alert. Used for calculations in the Value report.'
    }),
    type: 'number',
    category: [_constants.APP_ID],
    requiresPageReload: true,
    schema: _configSchema.schema.number(),
    solutionViews: ['classic', 'security']
  },
  [_constants.DEFAULT_VALUE_REPORT_RATE]: {
    name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultValueRateLabel', {
      defaultMessage: 'Value report analyst hourly rate'
    }),
    value: 75,
    description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultValueRateDescription', {
      defaultMessage: 'The average hourly rate for a security analyst. Used for calculations in the Value report.'
    }),
    type: 'number',
    category: [_constants.APP_ID],
    requiresPageReload: true,
    schema: _configSchema.schema.number(),
    solutionViews: ['classic', 'security']
  },
  [_constants.DEFAULT_VALUE_REPORT_TITLE]: {
    name: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultValueTitleLabel', {
      defaultMessage: 'Value report title'
    }),
    value: _i18n.i18n.translate('xpack.securitySolution.reports.uiSettings.defaultValueTitleTitle', {
      defaultMessage: 'Elastic AI value report'
    }),
    description: _i18n.i18n.translate('xpack.securitySolution.uiSettings.defaultValueTitleDescription', {
      defaultMessage: 'The title of the Value report.'
    }),
    type: 'string',
    category: [_constants.APP_ID],
    requiresPageReload: true,
    schema: _configSchema.schema.string(),
    solutionViews: ['classic', 'security']
  }
});
exports.getDefaultValueReportSettings = getDefaultValueReportSettings;