"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ruleParamsModifier = exports.deleteItemsFromArray = exports.addItemsToArray = void 0;
var _rule_management = require("../../../../../../common/api/detection_engine/rule_management");
var _invariant = require("../../../../../../common/utils/invariant");
var _utils = require("../../../rule_types/utils/utils");
var _constants = require("../../../../../../common/detection_engine/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const addItemsToArray = (arr, items) => Array.from(new Set([...arr, ...items]));
exports.addItemsToArray = addItemsToArray;
const deleteItemsFromArray = (arr, items) => {
  const itemsSet = new Set(items);
  return arr.filter(item => !itemsSet.has(item));
};

// Check if current params have a configured data view id
// and the action is not set to overwrite data views
exports.deleteItemsFromArray = deleteItemsFromArray;
const isDataViewExistsAndNotOverriden = (dataViewId, action) => dataViewId != null && !action.overwrite_data_views;

// Check if the index patterns added to the rule already exist in it
const hasIndexPatterns = (indexPatterns, action) => action.value.every(indexPattern => indexPatterns === null || indexPatterns === void 0 ? void 0 : indexPatterns.includes(indexPattern));

// Check if the index patterns to be deleted don't exist in the rule
const hasNotIndexPattern = (indexPatterns, action) => action.value.every(indexPattern => !(indexPatterns !== null && indexPatterns !== void 0 && indexPatterns.includes(indexPattern)));
const shouldSkipIndexPatternsBulkAction = (indexPatterns, dataViewId, action) => {
  if (isDataViewExistsAndNotOverriden(dataViewId, action)) {
    return true;
  }
  if (action.type === _rule_management.BulkActionEditTypeEnum.add_index_patterns) {
    return hasIndexPatterns(indexPatterns, action);
  }
  if (action.type === _rule_management.BulkActionEditTypeEnum.delete_index_patterns) {
    return hasNotIndexPattern(indexPatterns, action);
  }
  return false;
};

// Check if the investigation fields added to the rule already exist in it
const hasInvestigationFields = (investigationFields, action) => action.value.field_names.every(field => {
  var _investigationFields$;
  return (Array.isArray(investigationFields) ? investigationFields : (_investigationFields$ = investigationFields === null || investigationFields === void 0 ? void 0 : investigationFields.field_names) !== null && _investigationFields$ !== void 0 ? _investigationFields$ : []).includes(field);
});

// Check if the investigation fields to be deleted don't exist in the rule
const hasNoInvestigationFields = (investigationFields, action) => action.value.field_names.every(field => {
  var _investigationFields$2;
  return !(Array.isArray(investigationFields) ? investigationFields : (_investigationFields$2 = investigationFields === null || investigationFields === void 0 ? void 0 : investigationFields.field_names) !== null && _investigationFields$2 !== void 0 ? _investigationFields$2 : []).includes(field);
});
const shouldSkipInvestigationFieldsBulkAction = (investigationFields, action) => {
  if (action.type === _rule_management.BulkActionEditTypeEnum.add_investigation_fields) {
    return hasInvestigationFields(investigationFields, action);
  }
  if (action.type === _rule_management.BulkActionEditTypeEnum.delete_investigation_fields) {
    return hasNoInvestigationFields(investigationFields, action);
  }
  return false;
};
const hasMatchingDuration = (duration, actionDuration) => (duration === null || duration === void 0 ? void 0 : duration.value) === (actionDuration === null || actionDuration === void 0 ? void 0 : actionDuration.value) && (duration === null || duration === void 0 ? void 0 : duration.unit) === (actionDuration === null || actionDuration === void 0 ? void 0 : actionDuration.unit);
const shouldSkipAddAlertSuppressionBulkAction = (alertSuppression, action) => {
  if (!hasMatchingDuration(alertSuppression === null || alertSuppression === void 0 ? void 0 : alertSuppression.duration, action.value.duration)) {
    return false;
  }
  if ((alertSuppression === null || alertSuppression === void 0 ? void 0 : alertSuppression.missingFieldsStrategy) !== action.value.missing_fields_strategy) {
    return false;
  }
  return action.value.group_by.every(field => {
    var _alertSuppression$gro;
    return alertSuppression === null || alertSuppression === void 0 ? void 0 : (_alertSuppression$gro = alertSuppression.groupBy) === null || _alertSuppression$gro === void 0 ? void 0 : _alertSuppression$gro.includes(field);
  });
};

// eslint-disable-next-line complexity
const applyBulkActionEditToRuleParams = (existingRuleParams, action) => {
  let ruleParams = {
    ...existingRuleParams
  };
  // If the action is successfully applied and the rule params are modified,
  // we update the following flag to false. As soon as the current function
  // returns this flag as false, at least once, for any action, we know that
  // the rule needs to be marked as having its params updated.
  let isActionSkipped = false;
  switch (action.type) {
    // index_patterns actions
    // index pattern is not present in machine learning rule type, so we throw error on it
    case _rule_management.BulkActionEditTypeEnum.add_index_patterns:
      {
        var _ruleParams$index;
        (0, _invariant.invariant)(ruleParams.type !== 'machine_learning', "Index patterns can't be added. Machine learning rule doesn't have index patterns property");
        (0, _invariant.invariant)(ruleParams.type !== 'esql', "Index patterns can't be added. ES|QL rule doesn't have index patterns property");
        if (shouldSkipIndexPatternsBulkAction(ruleParams.index, ruleParams.dataViewId, action)) {
          isActionSkipped = true;
          break;
        }
        if (action.overwrite_data_views) {
          ruleParams.dataViewId = undefined;
        }
        ruleParams.index = addItemsToArray((_ruleParams$index = ruleParams.index) !== null && _ruleParams$index !== void 0 ? _ruleParams$index : [], action.value);
        break;
      }
    case _rule_management.BulkActionEditTypeEnum.delete_index_patterns:
      {
        (0, _invariant.invariant)(ruleParams.type !== 'machine_learning', "Index patterns can't be deleted. Machine learning rule doesn't have index patterns property");
        (0, _invariant.invariant)(ruleParams.type !== 'esql', "Index patterns can't be deleted. ES|QL rule doesn't have index patterns property");
        if (!action.overwrite_data_views && shouldSkipIndexPatternsBulkAction(ruleParams.index, ruleParams.dataViewId, action)) {
          isActionSkipped = true;
          break;
        }
        if (action.overwrite_data_views) {
          ruleParams.dataViewId = undefined;
        }
        if (ruleParams.index) {
          ruleParams.index = deleteItemsFromArray(ruleParams.index, action.value);
        }
        break;
      }
    case _rule_management.BulkActionEditTypeEnum.set_index_patterns:
      {
        (0, _invariant.invariant)(ruleParams.type !== 'machine_learning', "Index patterns can't be overwritten. Machine learning rule doesn't have index patterns property");
        (0, _invariant.invariant)(ruleParams.type !== 'esql', "Index patterns can't be overwritten. ES|QL rule doesn't have index patterns property");
        if (shouldSkipIndexPatternsBulkAction(ruleParams.index, ruleParams.dataViewId, action)) {
          isActionSkipped = true;
          break;
        }
        if (action.overwrite_data_views) {
          ruleParams.dataViewId = undefined;
        }
        ruleParams.index = action.value;
        break;
      }
    // investigation_fields actions
    case _rule_management.BulkActionEditTypeEnum.add_investigation_fields:
      {
        var _ref, _ruleParams$investiga;
        if (shouldSkipInvestigationFieldsBulkAction(ruleParams.investigationFields, action)) {
          isActionSkipped = true;
          break;
        }
        ruleParams.investigationFields = {
          field_names: addItemsToArray((_ref = Array.isArray(ruleParams.investigationFields) ? ruleParams.investigationFields : (_ruleParams$investiga = ruleParams.investigationFields) === null || _ruleParams$investiga === void 0 ? void 0 : _ruleParams$investiga.field_names) !== null && _ref !== void 0 ? _ref : [], action.value.field_names)
        };
        break;
      }
    case _rule_management.BulkActionEditTypeEnum.delete_investigation_fields:
      {
        if (shouldSkipInvestigationFieldsBulkAction(ruleParams.investigationFields, action)) {
          isActionSkipped = true;
          break;
        }
        if (ruleParams.investigationFields) {
          var _ref2, _ruleParams$investiga2;
          const fieldNames = deleteItemsFromArray((_ref2 = Array.isArray(ruleParams.investigationFields) ? ruleParams.investigationFields : (_ruleParams$investiga2 = ruleParams.investigationFields) === null || _ruleParams$investiga2 === void 0 ? void 0 : _ruleParams$investiga2.field_names) !== null && _ref2 !== void 0 ? _ref2 : [], action.value.field_names);
          ruleParams.investigationFields = fieldNames.length > 0 ? {
            field_names: fieldNames
          } : undefined;
        }
        break;
      }
    case _rule_management.BulkActionEditTypeEnum.set_investigation_fields:
      {
        if (shouldSkipInvestigationFieldsBulkAction(ruleParams.investigationFields, action)) {
          isActionSkipped = true;
          break;
        }
        ruleParams.investigationFields = action.value;
        break;
      }
    // alert suppression actions
    case _rule_management.BulkActionEditTypeEnum.delete_alert_suppression:
      {
        var _ruleParams;
        if (!((_ruleParams = ruleParams) !== null && _ruleParams !== void 0 && _ruleParams.alertSuppression)) {
          isActionSkipped = true;
          break;
        }
        ruleParams.alertSuppression = undefined;
        break;
      }
    case _rule_management.BulkActionEditTypeEnum.set_alert_suppression:
      {
        var _ruleParams2, _action$value$missing;
        (0, _invariant.invariant)(ruleParams.type !== 'threshold', "Threshold rule doesn't support this action. Use 'set_alert_suppression_for_threshold' action instead");
        if (shouldSkipAddAlertSuppressionBulkAction((_ruleParams2 = ruleParams) === null || _ruleParams2 === void 0 ? void 0 : _ruleParams2.alertSuppression, action)) {
          isActionSkipped = true;
          break;
        }
        ruleParams.alertSuppression = {
          groupBy: action.value.group_by,
          missingFieldsStrategy: (_action$value$missing = action.value.missing_fields_strategy) !== null && _action$value$missing !== void 0 ? _action$value$missing : _constants.DEFAULT_SUPPRESSION_MISSING_FIELDS_STRATEGY,
          duration: action.value.duration
        };
        break;
      }
    case _rule_management.BulkActionEditTypeEnum.set_alert_suppression_for_threshold:
      {
        var _ruleParams3, _ruleParams3$alertSup;
        (0, _invariant.invariant)(ruleParams.type === 'threshold', `${ruleParams.type} rule type doesn't support this action. Use 'set_alert_suppression' action instead.`);
        if (hasMatchingDuration((_ruleParams3 = ruleParams) === null || _ruleParams3 === void 0 ? void 0 : (_ruleParams3$alertSup = _ruleParams3.alertSuppression) === null || _ruleParams3$alertSup === void 0 ? void 0 : _ruleParams3$alertSup.duration, action.value.duration)) {
          isActionSkipped = true;
          break;
        }
        ruleParams.alertSuppression = {
          duration: action.value.duration
        };
        break;
      }
    // timeline actions
    case _rule_management.BulkActionEditTypeEnum.set_timeline:
      {
        ruleParams = {
          ...ruleParams,
          timelineId: action.value.timeline_id || undefined,
          timelineTitle: action.value.timeline_title || undefined
        };
        break;
      }
    // update look-back period in from and meta.from fields
    case _rule_management.BulkActionEditTypeEnum.set_schedule:
      {
        const from = (0, _utils.calculateFromValue)(action.value.interval, action.value.lookback);
        ruleParams = {
          ...ruleParams,
          meta: {
            ...ruleParams.meta,
            from: action.value.lookback
          },
          from
        };
        break;
      }
  }
  return {
    ruleParams,
    isActionSkipped
  };
};

/**
 * takes list of bulkEdit actions and apply them to rule.params by mutating it
 * @param existingRuleParams
 * @param actions
 * @returns mutated params, isParamsUpdateSkipped flag
 */
const ruleParamsModifier = (existingRuleParams, actions) => {
  let isParamsUpdateSkipped = true;
  const modifiedParams = actions.reduce((acc, action) => {
    const {
      ruleParams,
      isActionSkipped
    } = applyBulkActionEditToRuleParams(acc, action);

    // The rule was updated with at least one action, so mark our rule as updated
    if (!isActionSkipped) {
      isParamsUpdateSkipped = false;
    }
    return {
      ...acc,
      ...ruleParams
    };
  }, existingRuleParams);

  // increment version even if actions are empty, as attributes can be modified as well outside of ruleParamsModifier
  // version must not be modified for immutable rule. Otherwise prebuilt rules upgrade flow will be broken
  if (existingRuleParams.immutable === false) {
    modifiedParams.version += 1;
  }
  return {
    modifiedParams,
    isParamsUpdateSkipped
  };
};
exports.ruleParamsModifier = ruleParamsModifier;