"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNonMigratedSignalsInfo = exports.checkIfMigratedIndexOutdated = void 0;
var _get_index_versions_by_index = require("./get_index_versions_by_index");
var _get_signal_versions_by_index = require("./get_signal_versions_by_index");
var _helpers = require("./helpers");
var _get_latest_index_template_version = require("./get_latest_index_template_version");
var _get_index_alias_per_space = require("./get_index_alias_per_space");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const REINDEXED_PREFIX = '.reindexed-v8-';
const checkIfMigratedIndexOutdated = (indexName, indexVersionsByIndex, latestTemplateVersion) => {
  var _indexVersionsByIndex;
  const isIndexOutdated = (0, _helpers.isOutdated)({
    current: (_indexVersionsByIndex = indexVersionsByIndex[indexName]) !== null && _indexVersionsByIndex !== void 0 ? _indexVersionsByIndex : 0,
    target: latestTemplateVersion
  });
  if (!isIndexOutdated) {
    return false;
  }
  const nameWithoutPrefix = indexName.replace(REINDEXED_PREFIX, '.');
  const hasOutdatedMigratedIndices = Object.entries(indexVersionsByIndex).every(([index, version]) => {
    if (index === indexName) {
      return true;
    }
    if (index.startsWith(nameWithoutPrefix) || index.startsWith(indexName)) {
      return (0, _helpers.isOutdated)({
        current: version !== null && version !== void 0 ? version : 0,
        target: latestTemplateVersion
      });
    }
    return true;
  });
  return hasOutdatedMigratedIndices;
};
exports.checkIfMigratedIndexOutdated = checkIfMigratedIndexOutdated;
/**
 * gets lists of spaces and non-migrated signal indices
 */
const getNonMigratedSignalsInfo = async ({
  esClient,
  signalsIndex,
  logger
}) => {
  const signalsAliasAllSpaces = `${signalsIndex}-*`;
  try {
    const latestTemplateVersion = await (0, _get_latest_index_template_version.getLatestIndexTemplateVersion)({
      esClient,
      name: signalsAliasAllSpaces
    });
    const indexAliasesMap = await (0, _get_index_alias_per_space.getIndexAliasPerSpace)({
      esClient,
      signalsAliasAllSpaces,
      signalsIndex
    });
    const indices = Object.keys(indexAliasesMap);
    if (indices.length === 0) {
      return {
        isMigrationRequired: false,
        spaces: [],
        indices: []
      };
    }
    let indexVersionsByIndex = {};
    try {
      indexVersionsByIndex = await (0, _get_index_versions_by_index.getIndexVersionsByIndex)({
        esClient,
        index: indices
      });
    } catch (e) {
      logger.debug(`Getting information about legacy siem signals index version failed:"${e === null || e === void 0 ? void 0 : e.message}"`);
    }
    let signalVersionsByIndex = {};
    try {
      signalVersionsByIndex = await (0, _get_signal_versions_by_index.getSignalVersionsByIndex)({
        esClient,
        index: indices
      });
    } catch (e) {
      logger.debug(`Getting information about legacy siem signals versions failed:"${e === null || e === void 0 ? void 0 : e.message}"`);
    }
    const outdatedIndices = indices.reduce((acc, indexName) => {
      var _indexVersionsByIndex2, _signalVersionsByInde;
      const version = (_indexVersionsByIndex2 = indexVersionsByIndex[indexName]) !== null && _indexVersionsByIndex2 !== void 0 ? _indexVersionsByIndex2 : 0;
      const signalVersions = (_signalVersionsByInde = signalVersionsByIndex[indexName]) !== null && _signalVersionsByInde !== void 0 ? _signalVersionsByInde : [];

      // filter out migrated from 7.x to 8 indices
      if (indexName.startsWith(REINDEXED_PREFIX) && !checkIfMigratedIndexOutdated(indexName, indexVersionsByIndex, latestTemplateVersion)) {
        return acc;
      }
      const isOutdated = (0, _helpers.isOutdated)({
        current: version,
        target: latestTemplateVersion
      }) || (0, _helpers.signalsAreOutdated)({
        signalVersions,
        target: latestTemplateVersion
      });
      if (isOutdated) {
        acc.push({
          indexName,
          space: indexAliasesMap[indexName].space
        });
      }
      return acc;
    }, []);
    const outdatedIndexNames = outdatedIndices.map(outdatedIndex => outdatedIndex.indexName);

    // remove duplicated spaces
    const spaces = [...new Set(outdatedIndices.map(indexStatus => indexStatus.space))];
    const isMigrationRequired = outdatedIndices.length > 0;
    logger.debug(isMigrationRequired ? `Legacy siem signals indices require migration: "${outdatedIndexNames.join(', ')}" in "${spaces.join(', ')}" spaces` : 'No legacy siem indices require migration');
    return {
      isMigrationRequired,
      spaces,
      indices: outdatedIndexNames
    };
  } catch (e) {
    logger.debug(`Getting information about legacy siem signals failed:"${e === null || e === void 0 ? void 0 : e.message}"`);
    return {
      isMigrationRequired: false,
      spaces: [],
      indices: []
    };
  }
};
exports.getNonMigratedSignalsInfo = getNonMigratedSignalsInfo;