"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MAX_SCHEDULED_REPORT_LIST_SIZE = exports.DEFAULT_SCHEDULED_REPORT_LIST_SIZE = void 0;
exports.scheduledQueryFactory = scheduledQueryFactory;
exports.transformResponse = transformResponse;
exports.transformSingleResponse = transformSingleResponse;
var _reportingServer = require("@kbn/reporting-server");
var _rrule2 = require("@kbn/rrule");
var _spacesUtils = require("@kbn/spaces-utils");
var _result_type = require("@kbn/task-manager-plugin/server/lib/result_type");
var _saved_objects = require("../../../saved_objects");
var _audit_events = require("../audit_events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_SCHEDULED_REPORT_LIST_SIZE = exports.MAX_SCHEDULED_REPORT_LIST_SIZE = 100;
const DEFAULT_SCHEDULED_REPORT_LIST_SIZE = exports.DEFAULT_SCHEDULED_REPORT_LIST_SIZE = 10;
const SCHEDULED_REPORT_ID_FIELD = 'scheduled_report_id';
const CREATED_AT_FIELD = 'created_at';
const getUsername = user => user ? user.username : false;
const getEmptyApiResponse = (page, perPage) => ({
  page,
  per_page: perPage,
  total: 0,
  data: []
});
function transformSingleResponse(logger, so, lastResponse, nextRunResponse) {
  var _lastResponse$hits$hi, _lastRunForId$_source, _so$namespaces$, _so$namespaces;
  const id = so.id;
  const lastRunForId = ((_lastResponse$hits$hi = lastResponse === null || lastResponse === void 0 ? void 0 : lastResponse.hits.hits) !== null && _lastResponse$hits$hi !== void 0 ? _lastResponse$hits$hi : []).find(hit => {
    var _hit$fields, _hit$fields$SCHEDULED;
    return ((_hit$fields = hit.fields) === null || _hit$fields === void 0 ? void 0 : (_hit$fields$SCHEDULED = _hit$fields[SCHEDULED_REPORT_ID_FIELD]) === null || _hit$fields$SCHEDULED === void 0 ? void 0 : _hit$fields$SCHEDULED[0]) === id;
  });
  const nextRunForId = (nextRunResponse !== null && nextRunResponse !== void 0 ? nextRunResponse : []).find(taskOrError => (0, _result_type.isOk)(taskOrError) && taskOrError.value.id === id);
  let nextRun;
  if (!nextRunForId) {
    var _rrule$after;
    // try to calculate dynamically if we were not able to get from the task
    const schedule = so.attributes.schedule;

    // get start date
    let dtstart = new Date();
    const rruleStart = schedule.rrule.dtstart;
    if (rruleStart) {
      try {
        // if start date is provided and in the future, use it, otherwise use current time
        const startDateValue = new Date(rruleStart).valueOf();
        const now = Date.now();
        if (startDateValue > now) {
          dtstart = new Date(startDateValue + 60000); // add 1 minute to ensure it's in the future
        }
      } catch (e) {
        logger.debug(`Failed to parse rrule.dtstart for scheduled report next run calculation - default to now ${id}: ${e.message}`);
      }
    }
    const _rrule = new _rrule2.RRule({
      ...schedule.rrule,
      dtstart
    });
    nextRun = (_rrule$after = _rrule.after(new Date())) === null || _rrule$after === void 0 ? void 0 : _rrule$after.toISOString();
  } else {
    nextRun = (0, _result_type.isOk)(nextRunForId) ? nextRunForId.value.runAt.toISOString() : undefined;
  }
  let payload;
  try {
    payload = JSON.parse(so.attributes.payload);
  } catch (e) {
    logger.warn(`Failed to parse payload for scheduled report ${id}: ${e.message}`);
  }
  return {
    id,
    created_at: so.attributes.createdAt,
    created_by: so.attributes.createdBy,
    enabled: so.attributes.enabled,
    jobtype: so.attributes.jobType,
    last_run: lastRunForId === null || lastRunForId === void 0 ? void 0 : (_lastRunForId$_source = lastRunForId._source) === null || _lastRunForId$_source === void 0 ? void 0 : _lastRunForId$_source[CREATED_AT_FIELD],
    next_run: nextRun,
    notification: so.attributes.notification,
    payload,
    schedule: so.attributes.schedule,
    space_id: (_so$namespaces$ = (_so$namespaces = so.namespaces) === null || _so$namespaces === void 0 ? void 0 : _so$namespaces[0]) !== null && _so$namespaces$ !== void 0 ? _so$namespaces$ : _spacesUtils.DEFAULT_SPACE_ID,
    title: so.attributes.title
  };
}
function transformResponse(logger, result, lastResponse, nextRunResponse) {
  return {
    page: result.page,
    per_page: result.per_page,
    total: result.total,
    data: result.saved_objects.map(so => transformSingleResponse(logger, so, lastResponse, nextRunResponse))
  };
}
function scheduledQueryFactory(reportingCore) {
  return {
    async list(logger, req, res, user, page = 1, size = DEFAULT_SCHEDULED_REPORT_LIST_SIZE) {
      try {
        const esClient = await reportingCore.getEsClient();
        const auditLogger = await reportingCore.getAuditLogger(req);
        const savedObjectsClient = await reportingCore.getScopedSoClient(req);
        const taskManager = await reportingCore.getTaskManager();
        const username = getUsername(user);

        // if user has Manage Reporting privileges, we can list
        // scheduled reports for all users in this space, otherwise
        // we will filter only to the scheduled reports created by the user
        const canManageReporting = await reportingCore.canManageReportingForSpace(req);
        const response = await savedObjectsClient.find({
          type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
          page,
          perPage: size,
          ...(!canManageReporting ? {
            filter: `scheduled_report.attributes.createdBy: "${username}"`
          } : {})
        });
        if (!response) {
          return getEmptyApiResponse(page, size);
        }
        const scheduledReportIdsAndName = response === null || response === void 0 ? void 0 : response.saved_objects.map(so => ({
          id: so.id,
          name: so.attributes.title
        }));
        if (!scheduledReportIdsAndName || scheduledReportIdsAndName.length === 0) {
          return getEmptyApiResponse(page, size);
        }
        scheduledReportIdsAndName.forEach(({
          id,
          name
        }) => auditLogger.log((0, _audit_events.scheduledReportAuditEvent)({
          action: _audit_events.ScheduledReportAuditAction.LIST,
          savedObject: {
            type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
            id,
            name
          }
        })));
        const scheduledReportIds = scheduledReportIdsAndName.map(({
          id
        }) => id);
        let lastRunResponse;
        try {
          lastRunResponse = await esClient.asInternalUser.search({
            index: _reportingServer.REPORTING_DATA_STREAM_WILDCARD_WITH_LEGACY,
            size,
            _source: [CREATED_AT_FIELD],
            sort: [{
              [CREATED_AT_FIELD]: {
                order: 'desc'
              }
            }],
            query: {
              bool: {
                filter: [{
                  terms: {
                    [SCHEDULED_REPORT_ID_FIELD]: scheduledReportIds
                  }
                }]
              }
            },
            collapse: {
              field: SCHEDULED_REPORT_ID_FIELD
            }
          });
        } catch (error) {
          // if no scheduled reports have run yet, we will get an error from the collapse query
          // ignore these and return an empty last run
          logger.warn(`Error getting last run for scheduled reports: ${error.message}`);
        }
        let nextRunResponse;
        try {
          nextRunResponse = await taskManager.bulkGet(scheduledReportIds);
        } catch (error) {
          // swallow this error
          logger.warn(`Error getting next run for scheduled reports: ${error.message}`);
        }
        return transformResponse(logger, response, lastRunResponse, nextRunResponse);
      } catch (error) {
        throw res.customError({
          statusCode: 500,
          body: `Error listing scheduled reports: ${error.message}`
        });
      }
    },
    async bulkDisable(logger, req, res, ids, user) {
      try {
        const savedObjectsClient = await reportingCore.getScopedSoClient(req);
        const taskManager = await reportingCore.getTaskManager();
        const auditLogger = await reportingCore.getAuditLogger(req);
        const bulkErrors = [];
        const disabledScheduledReportIds = new Set();
        let taskIdsToDisable = [];
        const username = getUsername(user);

        // if user has Manage Reporting privileges, they can disable
        // scheduled reports for all users in this space
        const canManageReporting = await reportingCore.canManageReportingForSpace(req);
        const bulkGetResult = await savedObjectsClient.bulkGet(ids.map(id => ({
          id,
          type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE
        })));
        const scheduledReportSavedObjectsToUpdate = [];
        for (const so of bulkGetResult.saved_objects) {
          if (so.error) {
            bulkErrors.push({
              message: so.error.message,
              status: so.error.statusCode,
              id: so.id
            });
          } else {
            // check if user is allowed to update this scheduled report
            if (so.attributes.createdBy !== username && !canManageReporting) {
              var _so$attributes;
              bulkErrors.push({
                message: `Not found.`,
                status: 404,
                id: so.id
              });
              logger.warn(`User "${username}" attempted to disable scheduled report "${so.id}" created by "${so.attributes.createdBy}" without sufficient privileges.`);
              auditLogger.log((0, _audit_events.scheduledReportAuditEvent)({
                action: _audit_events.ScheduledReportAuditAction.DISABLE,
                savedObject: {
                  type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
                  id: so.id,
                  name: so === null || so === void 0 ? void 0 : (_so$attributes = so.attributes) === null || _so$attributes === void 0 ? void 0 : _so$attributes.title
                },
                error: new Error(`Not found.`)
              }));
            } else if (so.attributes.enabled === false) {
              logger.debug(`Scheduled report ${so.id} is already disabled`);
              disabledScheduledReportIds.add(so.id);
            } else {
              auditLogger.log((0, _audit_events.scheduledReportAuditEvent)({
                action: _audit_events.ScheduledReportAuditAction.DISABLE,
                savedObject: {
                  type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
                  id: so.id,
                  name: so.attributes.title
                },
                outcome: 'unknown'
              }));
              scheduledReportSavedObjectsToUpdate.push(so);
            }
          }
        }

        // nothing to update, return early
        if (scheduledReportSavedObjectsToUpdate.length > 0) {
          const bulkUpdateResult = await savedObjectsClient.bulkUpdate(scheduledReportSavedObjectsToUpdate.map(so => ({
            id: so.id,
            type: so.type,
            attributes: {
              enabled: false
            }
          })));
          for (const so of bulkUpdateResult.saved_objects) {
            if (so.error) {
              var _so$attributes2;
              bulkErrors.push({
                message: so.error.message,
                status: so.error.statusCode,
                id: so.id
              });
              auditLogger.log((0, _audit_events.scheduledReportAuditEvent)({
                action: _audit_events.ScheduledReportAuditAction.DISABLE,
                savedObject: {
                  type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
                  id: so.id,
                  name: so === null || so === void 0 ? void 0 : (_so$attributes2 = so.attributes) === null || _so$attributes2 === void 0 ? void 0 : _so$attributes2.title
                },
                error: new Error(so.error.message)
              }));
            } else {
              taskIdsToDisable.push(so.id);
            }
          }
        } else {
          return {
            scheduled_report_ids: [...disabledScheduledReportIds],
            errors: bulkErrors,
            total: disabledScheduledReportIds.size + bulkErrors.length
          };
        }

        // it's possible that the scheduled_report saved object was disabled but
        // task disabling failed so add the list of already disabled IDs
        // task manager filters out disabled tasks so this will not cause extra load
        taskIdsToDisable = taskIdsToDisable.concat([...disabledScheduledReportIds]);
        const resultFromDisablingTasks = await taskManager.bulkDisable(taskIdsToDisable);
        for (const error of resultFromDisablingTasks.errors) {
          bulkErrors.push({
            message: `Scheduled report disabled but task disabling failed due to: ${error.error.message}`,
            status: error.error.statusCode,
            id: error.id
          });
        }
        for (const result of resultFromDisablingTasks.tasks) {
          disabledScheduledReportIds.add(result.id);
        }
        return {
          scheduled_report_ids: [...disabledScheduledReportIds],
          errors: bulkErrors,
          total: disabledScheduledReportIds.size + bulkErrors.length
        };
      } catch (error) {
        throw res.customError({
          statusCode: 500,
          body: `Error disabling scheduled reports: ${error.message}`
        });
      }
    }
  };
}