"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerMCPRoutes = registerMCPRoutes;
var _mcp = require("@modelcontextprotocol/sdk/server/mcp.js");
var _types = require("@modelcontextprotocol/sdk/types.js");
var _configSchema = require("@kbn/config-schema");
var _nodePath = _interopRequireDefault(require("node:path"));
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _features = require("../../common/features");
var _wrap_handler = require("./wrap_handler");
var _kibana_mcp_http_transport = require("../utils/mcp/kibana_mcp_http_transport");
var _mcp2 = require("../../common/mcp");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MCP_SERVER_NAME = 'elastic-mcp-server';
const MCP_SERVER_VERSION = '0.0.1';
function registerMCPRoutes({
  router,
  getInternalServices,
  logger
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });
  router.versioned.post({
    path: _mcp2.MCP_SERVER_PATH,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'MCP server',
    description: 'WARNING: This endpoint is designed for MCP clients (Claude Desktop, Cursor, VS Code, etc.) and should not be used directly via REST APIs. Use MCP Inspector or native MCP clients instead.',
    options: {
      tags: ['mcp', 'oas-tag:agent builder'],
      xsrfRequired: false,
      availability: {
        stability: 'experimental',
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: _configSchema.schema.object({}, {
          unknowns: 'allow',
          meta: {
            description: 'JSON-RPC 2.0 request payload for MCP server communication.'
          }
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/mcp_initialize.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    let transport;
    let server;
    try {
      transport = new _kibana_mcp_http_transport.KibanaMcpHttpTransport({
        sessionIdGenerator: undefined,
        logger
      });

      // Instantiate new MCP server upon every request, no session persistence
      server = new _mcp.McpServer({
        name: MCP_SERVER_NAME,
        version: MCP_SERVER_VERSION
      });
      const {
        tools: toolService
      } = getInternalServices();
      const registry = await toolService.getRegistry({
        request
      });
      const tools = await registry.list({});
      const idMapping = (0, _langchain.createToolIdMappings)(tools);

      // Expose tools scoped to the request
      for (const tool of tools) {
        var _idMapping$get;
        const toolSchema = await tool.getSchema();
        server.tool((_idMapping$get = idMapping.get(tool.id)) !== null && _idMapping$get !== void 0 ? _idMapping$get : tool.id, tool.description, toolSchema.shape, async args => {
          const toolResult = await registry.execute({
            toolId: tool.id,
            toolParams: args
          });
          return {
            content: [{
              type: 'text',
              text: JSON.stringify(toolResult)
            }]
          };
        });
      }
      request.events.aborted$.subscribe(async () => {
        var _transport, _server;
        await ((_transport = transport) === null || _transport === void 0 ? void 0 : _transport.close().catch(error => {
          logger.error('MCP Server: Error closing transport', {
            error
          });
        }));
        await ((_server = server) === null || _server === void 0 ? void 0 : _server.close().catch(error => {
          logger.error('MCP Server: Error closing server', {
            error
          });
        }));
      });
      await server.connect(transport);
      return await transport.handleRequest(request, response);
    } catch (error) {
      logger.error('MCP Server: Error handling request', {
        error
      });
      try {
        var _transport2;
        await ((_transport2 = transport) === null || _transport2 === void 0 ? void 0 : _transport2.close());
      } catch (closeError) {
        logger.error('MCP Server: Error closing transport during error handling', {
          error: closeError
        });
      }
      if (server) {
        try {
          await server.close();
        } catch (closeError) {
          logger.error('MCP Server: Error closing server during error handling', {
            error: closeError
          });
        }
      }
      logger.error('MCP Server: Error handling request', {
        error
      });
      return response.customError({
        statusCode: 500,
        body: {
          message: `Internal server error: ${error}`,
          attributes: {
            code: _types.ErrorCode.InternalError
          }
        }
      });
    }
  }));
}