"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.columnExists = exports.buildUserDefinedColumnsDefinitions = void 0;
exports.createInferenceEndpointToCompletionItem = createInferenceEndpointToCompletionItem;
exports.findFinalWord = void 0;
exports.findPreviousWord = findPreviousWord;
exports.getControlSuggestion = getControlSuggestion;
exports.getControlSuggestionIfSupported = getControlSuggestionIfSupported;
exports.getExpressionPosition = void 0;
exports.getFieldsOrFunctionsSuggestions = getFieldsOrFunctionsSuggestions;
exports.getFragmentData = getFragmentData;
exports.getLastNonWhitespaceChar = getLastNonWhitespaceChar;
exports.getLookupIndexCreateSuggestion = getLookupIndexCreateSuggestion;
exports.getSafeInsertText = void 0;
exports.getValidSignaturesAndTypesToSuggestNext = getValidSignaturesAndTypesToSuggestNext;
exports.handleFragment = handleFragment;
exports.pushItUpInTheList = pushItUpInTheList;
exports.shouldBeQuotedText = void 0;
exports.suggestForExpression = suggestForExpression;
exports.withAutoSuggest = withAutoSuggest;
exports.within = void 0;
exports.withinQuotes = withinQuotes;
var _i18n = require("@kbn/i18n");
var _esqlTypes = require("@kbn/esql-types");
var _lodash = require("lodash");
var _types = require("../../../commands_registry/types");
var _literals = require("../literals");
var _constants = require("../../constants");
var _types2 = require("../../types");
var _shared = require("../shared");
var _expressions = require("../expressions");
var _functions = require("../functions");
var _all_operators = require("../../all_operators");
var _operators = require("../operators");
var _is = require("../../../ast/is");
var _walker = require("../../../walker");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const within = (position, location) => Boolean(location && location.min <= position && location.max >= position);
exports.within = within;
const shouldBeQuotedText = (text, {
  dashSupported
} = {}) => {
  return dashSupported ? /[^a-zA-Z\d_\.@-]/.test(text) : /[^a-zA-Z\d_\.@]/.test(text);
};
exports.shouldBeQuotedText = shouldBeQuotedText;
const getSafeInsertText = (text, options = {}) => {
  return shouldBeQuotedText(text, options) ? `\`${text.replace(/`/g, '``')}\`` : text;
};
exports.getSafeInsertText = getSafeInsertText;
const buildUserDefinedColumnsDefinitions = userDefinedColumns => userDefinedColumns.map(label => ({
  label,
  text: getSafeInsertText(label),
  kind: 'Variable',
  detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.variableDefinition', {
    defaultMessage: `Column specified by the user within the ES|QL query`
  }),
  sortText: 'D'
}));
exports.buildUserDefinedColumnsDefinitions = buildUserDefinedColumnsDefinitions;
function pushItUpInTheList(suggestions, shouldPromote) {
  if (!shouldPromote) {
    return suggestions;
  }
  return suggestions.map(({
    sortText,
    ...rest
  }) => ({
    ...rest,
    sortText: `1${sortText}`
  }));
}
const findFinalWord = text => {
  const words = text.split(/\s+/);
  return words[words.length - 1];
};
exports.findFinalWord = findFinalWord;
function findPreviousWord(text) {
  const words = text.split(/\s+/);
  return words[words.length - 2];
}
function withinQuotes(text) {
  const quoteCount = (text.match(/"/g) || []).length;
  return quoteCount % 2 === 1;
}

/**
 * This function handles the logic to suggest completions
 * for a given fragment of text in a generic way. A good example is
 * a field name.
 *
 * When typing a field name, there are 2 scenarios
 *
 * 1. field name is incomplete (includes the empty string)
 * KEEP /
 * KEEP fie/
 *
 * 2. field name is complete
 * KEEP field/
 *
 * This function provides a framework for detecting and handling both scenarios in a clean way.
 *
 * @param innerText - the query text before the current cursor position
 * @param isFragmentComplete — return true if the fragment is complete
 * @param getSuggestionsForIncomplete — gets suggestions for an incomplete fragment
 * @param getSuggestionsForComplete - gets suggestions for a complete fragment
 * @returns
 */
function handleFragment(innerText, isFragmentComplete, getSuggestionsForIncomplete, getSuggestionsForComplete) {
  const {
    fragment,
    rangeToReplace
  } = getFragmentData(innerText);
  if (!fragment) {
    return getSuggestionsForIncomplete('');
  } else {
    if (isFragmentComplete(fragment)) {
      return getSuggestionsForComplete(fragment, rangeToReplace);
    } else {
      return getSuggestionsForIncomplete(fragment, rangeToReplace);
    }
  }
}
function getFragmentData(innerText) {
  const fragment = findFinalWord(innerText);
  if (!fragment) {
    return {
      fragment: '',
      rangeToReplace: {
        start: 0,
        end: 0
      }
    };
  } else {
    const rangeToReplace = {
      start: innerText.length - fragment.length,
      end: innerText.length
    };
    return {
      fragment,
      rangeToReplace
    };
  }
}

/**
 * TODO — split this into distinct functions, one for fields, one for functions, one for literals
 */
async function getFieldsOrFunctionsSuggestions(types, location, getFieldsByType, {
  functions,
  columns: fields,
  values = false,
  literals = false
}, {
  ignoreFn = [],
  ignoreColumns = []
} = {}, hasMinimumLicenseRequired, activeProduct) {
  const filteredFieldsByType = pushItUpInTheList(await (fields ? getFieldsByType(types, ignoreColumns, {
    advanceCursor: location === _types.Location.SORT,
    openSuggestions: location === _types.Location.SORT,
    variableType: values ? _esqlTypes.ESQLVariableType.VALUES : _esqlTypes.ESQLVariableType.FIELDS
  }) : []), functions);

  // could also be in stats (bucket) but our autocomplete is not great yet
  const displayDateSuggestions = types.includes('date') && [_types.Location.WHERE, _types.Location.EVAL].includes(location);
  const suggestions = filteredFieldsByType.concat(displayDateSuggestions ? (0, _literals.getDateLiterals)() : [], functions ? (0, _functions.getFunctionSuggestions)({
    location,
    returnTypes: types,
    ignored: ignoreFn
  }, hasMinimumLicenseRequired, activeProduct) : [], literals ? (0, _literals.getCompatibleLiterals)(types) : []);
  return suggestions;
}
function getLastNonWhitespaceChar(text) {
  return text[text.trimEnd().length - 1];
}
const columnExists = (col, context) => Boolean(context ? (0, _shared.getColumnByName)(col, context) : undefined);

/**
 * The position of the cursor within an expression.
 */
exports.columnExists = columnExists;
/**
 * Escapes special characters in a string to be used as a literal match in a regular expression.
 * @param {string} text The input string to escape.
 * @returns {string} The escaped string.
 */
function escapeRegExp(text) {
  // Characters with special meaning in regex: . * + ? ^ $ { } ( ) | [ ] \
  // We need to escape all of them. The `$&` in the replacement string means "the matched substring".
  return text.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

/**
 * Determines the position of the cursor within an expression.
 * @param innerText
 * @param expressionRoot
 * @returns
 */
const getExpressionPosition = (innerText, expressionRoot) => {
  const endsWithNot = / not$/i.test(innerText.trimEnd());
  if (endsWithNot && !(expressionRoot && (0, _is.isFunctionExpression)(expressionRoot) &&
  // See https://github.com/elastic/kibana/issues/199401
  // for more information on this check...
  ['is null', 'is not null'].includes(expressionRoot.name))) {
    return 'after_not';
  }
  if (expressionRoot) {
    if ((0, _is.isColumn)(expressionRoot) &&
    // and not directly after the column name or prefix e.g. "colu/"
    // we are escaping the column name here as it may contain special characters such as ??
    !new RegExp(`${escapeRegExp(expressionRoot.parts.join('\\.'))}$`).test(innerText)) {
      return 'after_column';
    }
    if ((0, _is.isFunctionExpression)(expressionRoot) && expressionRoot.subtype === 'variadic-call') {
      return 'after_function';
    }
    if ((0, _is.isFunctionExpression)(expressionRoot) && expressionRoot.subtype !== 'variadic-call') {
      return 'after_operator';
    }
    if ((0, _is.isLiteral)(expressionRoot)) {
      return 'after_literal';
    }
  }
  return 'empty_expression';
};

/**
 * Creates suggestion within an expression.
 *
 * TODO — should this function know about the command context
 * or would we prefer a set of generic configuration options?
 *
 * @param param0
 * @returns
 */
exports.getExpressionPosition = getExpressionPosition;
async function suggestForExpression({
  expressionRoot,
  innerText,
  getColumnsByType: _getColumnsByType,
  location,
  preferredExpressionType,
  context,
  advanceCursorAfterInitialColumn = true,
  hasMinimumLicenseRequired,
  activeProduct,
  ignoredColumnsForEmptyExpression = []
}) {
  const getColumnsByType = _getColumnsByType ? _getColumnsByType : () => Promise.resolve([]);
  const suggestions = [];
  const position = getExpressionPosition(innerText, expressionRoot);
  switch (position) {
    /**
     * After a literal, column, or complete (non-operator) function call
     */
    case 'after_literal':
    case 'after_column':
    case 'after_function':
      const expressionType = (0, _expressions.getExpressionType)(expressionRoot, context === null || context === void 0 ? void 0 : context.columns);
      if (!(0, _types2.isParameterType)(expressionType)) {
        break;
      }
      suggestions.push(...(0, _operators.getOperatorSuggestions)({
        location,
        // In case of a param literal, we don't know the type of the left operand
        // so we can only suggest operators that accept any type as a left operand
        leftParamType: (0, _shared.isParamExpressionType)(expressionType) ? undefined : expressionType,
        ignored: ['='],
        allowed: expressionType === 'boolean' && position === 'after_literal' ? [..._all_operators.logicalOperators.filter(({
          locationsAvailable
        }) => locationsAvailable.includes(location)).map(({
          name
        }) => name)] : undefined
      }, hasMinimumLicenseRequired, activeProduct));
      break;

    /**
     * After a NOT keyword
     *
     * the NOT function is a special operator that can be used in different ways,
     * and not all these are mapped within the AST data structure: in particular
     * <COMMAND> <field> NOT <here>
     * is an incomplete statement and it results in a missing AST node, so we need to detect
     * from the query string itself
     *
     * (this comment was copied but seems to still apply)
     */
    case 'after_not':
      if (expressionRoot && (0, _is.isFunctionExpression)(expressionRoot) && expressionRoot.name === 'not') {
        suggestions.push(...(0, _functions.getFunctionSuggestions)({
          location,
          returnTypes: ['boolean']
        }, hasMinimumLicenseRequired, activeProduct), ...(await getColumnsByType('boolean', [], {
          advanceCursor: true,
          openSuggestions: true
        })));
      } else {
        suggestions.push(...(0, _operators.getOperatorsSuggestionsAfterNot)());
      }
      break;

    /**
     * After an operator (e.g. AND, OR, IS NULL, +, etc.)
     */
    case 'after_operator':
      if (!expressionRoot) {
        break;
      }
      if (!(0, _is.isFunctionExpression)(expressionRoot) || expressionRoot.subtype === 'variadic-call') {
        // this is already guaranteed in the getPosition function, but TypeScript doesn't know
        break;
      }
      let rightmostOperator = expressionRoot;
      // get rightmost function
      const walker = new _walker.Walker({
        visitFunction: fn => {
          if (fn.location.min > rightmostOperator.location.min && fn.subtype !== 'variadic-call') rightmostOperator = fn;
        }
      });
      walker.walkFunction(expressionRoot);

      // See https://github.com/elastic/kibana/issues/199401 for an explanation of
      // why this check has to be so convoluted
      if (rightmostOperator.text.toLowerCase().trim().endsWith('null')) {
        suggestions.push(..._all_operators.logicalOperators.map(_operators.getOperatorSuggestion));
        break;
      }
      suggestions.push(...(await (0, _operators.getSuggestionsToRightOfOperatorExpression)({
        queryText: innerText,
        location,
        rootOperator: rightmostOperator,
        preferredExpressionType,
        getExpressionType: expression => (0, _expressions.getExpressionType)(expression, context === null || context === void 0 ? void 0 : context.columns),
        getColumnsByType,
        hasMinimumLicenseRequired,
        activeProduct
      })));
      break;
    case 'empty_expression':
      const columnSuggestions = await getColumnsByType('any', ignoredColumnsForEmptyExpression, {
        advanceCursor: advanceCursorAfterInitialColumn,
        openSuggestions: true
      });
      suggestions.push(...pushItUpInTheList(columnSuggestions, true), ...(0, _functions.getFunctionSuggestions)({
        location
      }, hasMinimumLicenseRequired, activeProduct));
      break;
  }

  /**
   * Attach replacement ranges if there's a prefix.
   *
   * Can't rely on Monaco because
   * - it counts "." as a word separator
   * - it doesn't handle multi-word completions (like "is null")
   *
   * TODO - think about how to generalize this — issue: https://github.com/elastic/kibana/issues/209905
   */
  const hasNonWhitespacePrefix = !/\s/.test(innerText[innerText.length - 1]);
  suggestions.forEach(s => {
    if (['IS NULL', 'IS NOT NULL'].includes(s.text)) {
      // this suggestion has spaces in it (e.g. "IS NOT NULL")
      // so we need to see if there's an overlap
      s.rangeToReplace = (0, _shared.getOverlapRange)(innerText, s.text);
      return;
    } else if (hasNonWhitespacePrefix) {
      // get index of first char of final word
      const lastNonWhitespaceIndex = innerText.search(/\S(?=\S*$)/);
      s.rangeToReplace = {
        start: lastNonWhitespaceIndex,
        end: innerText.length
      };
    }
  });
  return suggestions;
}
function getControlSuggestion(type, variables) {
  return [{
    label: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.createControlLabel', {
      defaultMessage: 'Create control'
    }),
    text: '',
    kind: 'Issue',
    detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.createControlDetailLabel', {
      defaultMessage: 'Click to create'
    }),
    sortText: '1',
    command: {
      id: `esql.control.${type}.create`,
      title: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.createControlDetailLabel', {
        defaultMessage: 'Click to create'
      })
    }
  }, ...(variables !== null && variables !== void 0 && variables.length ? (0, _literals.buildConstantsDefinitions)(variables, _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.namedParamDefinition', {
    defaultMessage: 'Named parameter'
  }), '1A') : [])];
}
const getVariablePrefix = variableType => variableType === _esqlTypes.ESQLVariableType.FIELDS || variableType === _esqlTypes.ESQLVariableType.FUNCTIONS ? '??' : '?';
function getControlSuggestionIfSupported(supportsControls, type, variables, shouldBePrefixed = true) {
  var _variables$filter;
  if (!supportsControls) {
    return [];
  }
  const prefix = shouldBePrefixed ? getVariablePrefix(type) : '';
  const filteredVariables = (_variables$filter = variables === null || variables === void 0 ? void 0 : variables.filter(variable => variable.type === type)) !== null && _variables$filter !== void 0 ? _variables$filter : [];
  const controlSuggestion = getControlSuggestion(type, filteredVariables === null || filteredVariables === void 0 ? void 0 : filteredVariables.map(v => `${prefix}${v.key}`));
  return controlSuggestion;
}
function getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex) {
  // Filter down to signatures that match every params up to the current argIndex
  // e.g. BUCKET(longField, /) => all signatures with first param as long column type
  // or BUCKET(longField, 2, /) => all signatures with (longField, integer, ...)
  const relevantFuncSignatures = fnDefinition.signatures.filter(s => {
    var _s$params;
    return ((_s$params = s.params) === null || _s$params === void 0 ? void 0 : _s$params.length) >= argIndex && s.params.slice(0, argIndex).every(({
      type: dataType
    }, idx) => (0, _expressions.argMatchesParamType)(enrichedArgs[idx].dataType, dataType, (0, _is.isLiteral)(enrichedArgs[idx]), true));
  });
  return relevantFuncSignatures;
}

/**
 * Given a function signature, returns the compatible types to suggest for the next argument
 *
 * @param fnDefinition: the function definition
 * @param enrichedArgs: AST args with enriched esType info to match with function signatures
 * @param argIndex: the index of the argument to suggest for
 * @returns
 */
function getCompatibleTypesToSuggestNext(fnDefinition, enrichedArgs, argIndex) {
  // First, narrow down to valid function signatures based on previous arguments
  const relevantFuncSignatures = getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex);

  // Then, get the compatible types to suggest for the next argument
  const compatibleTypesToSuggestForArg = (0, _lodash.uniqBy)(relevantFuncSignatures.map(f => f.params[argIndex]).filter(d => d), o => `${o.type}-${o.constantOnly}`);
  return compatibleTypesToSuggestForArg;
}

/**
 * Given a function signature, returns the parameter at the given position, even if it's undefined or null
 *
 * @param {params}
 * @param position
 * @returns
 */
function strictlyGetParamAtPosition({
  params
}, position) {
  return params[position] ? params[position] : null;
}
function getValidSignaturesAndTypesToSuggestNext(node, context, fnDefinition, fullText, offset) {
  const argTypes = node.args.map(arg => (0, _expressions.getExpressionType)(arg, context === null || context === void 0 ? void 0 : context.columns));
  const enrichedArgs = node.args.map((arg, idx) => ({
    ...arg,
    dataType: argTypes[idx]
  }));

  // pick the type of the next arg
  const shouldGetNextArgument = node.text.includes(_constants.EDITOR_MARKER);
  let argIndex = Math.max(node.args.length, 0);
  if (!shouldGetNextArgument && argIndex) {
    argIndex -= 1;
  }
  const validSignatures = getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex);
  // Retrieve unique of types that are compatiable for the current arg
  const typesToSuggestNext = getCompatibleTypesToSuggestNext(fnDefinition, enrichedArgs, argIndex);
  const hasMoreMandatoryArgs = !validSignatures
  // Types available to suggest next after this argument is completed
  .map(signature => strictlyGetParamAtPosition(signature, argIndex + 1))
  // when a param is null, it means param is optional
  // If there's at least one param that is optional, then
  // no need to suggest comma
  .some(p => p === null || (p === null || p === void 0 ? void 0 : p.optional) === true);

  // Whether to prepend comma to suggestion string
  // E.g. if true, "fieldName" -> "fieldName, "
  const alreadyHasComma = fullText ? fullText[offset] === ',' : false;
  const shouldAddComma = hasMoreMandatoryArgs && fnDefinition.type !== _types2.FunctionDefinitionTypes.OPERATOR && !alreadyHasComma;
  const currentArg = enrichedArgs[argIndex];
  return {
    shouldAddComma,
    typesToSuggestNext,
    validSignatures,
    hasMoreMandatoryArgs,
    enrichedArgs,
    argIndex,
    currentArg
  };
}
function getLookupIndexCreateSuggestion(innerText, indexName) {
  const start = indexName ? innerText.lastIndexOf(indexName) : -1;
  const rangeToReplace = indexName && start !== -1 ? {
    start,
    end: start + indexName.length
  } : undefined;
  return {
    label: indexName ? _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createLookupIndexWithName', {
      defaultMessage: 'Create lookup index "{indexName}"',
      values: {
        indexName
      }
    }) : _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createLookupIndex', {
      defaultMessage: 'Create lookup index'
    }),
    text: indexName,
    kind: 'Issue',
    filterText: indexName,
    detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createLookupIndexDetailLabel', {
      defaultMessage: 'Click to create'
    }),
    sortText: '1A',
    command: {
      id: `esql.lookup_index.create`,
      title: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createLookupIndexDetailLabel', {
        defaultMessage: 'Click to create'
      }),
      arguments: [{
        indexName
      }]
    },
    rangeToReplace,
    incomplete: true
  };
}
function createInferenceEndpointToCompletionItem(inferenceEndpoint) {
  return {
    detail: _i18n.i18n.translate('kbn-esql-ast.esql.definitions.rerankInferenceIdDoc', {
      defaultMessage: 'Inference endpoint used for the completion'
    }),
    kind: 'Reference',
    label: inferenceEndpoint.inference_id,
    sortText: '1',
    text: inferenceEndpoint.inference_id
  };
}

/**
 * Given a suggestion item, decorates it with editor.action.triggerSuggest
 * that triggers the autocomplete dialog again after accepting the suggestion.
 *
 * If the suggestion item already has a custom command, it will preserve it.
 */
function withAutoSuggest(suggestionItem) {
  return {
    ...suggestionItem,
    command: suggestionItem.command ? suggestionItem.command : {
      title: 'Trigger Suggestion Dialog',
      id: 'editor.action.triggerSuggest'
    }
  };
}