"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.selectHasUnsavedChanges = void 0;
var _public = require("@kbn/saved-search-plugin/public");
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _unifiedHistogram = require("@kbn/unified-histogram");
var _unifiedDataTable = require("@kbn/unified-data-table");
var _discover_app_state_container = require("../../discover_app_state_container");
var _add_log = require("../../../../../utils/add_log");
var _tabs = require("./tabs");
var _runtime_state = require("../runtime_state");
var _tab_mapping_utils = require("../tab_mapping_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const selectHasUnsavedChanges = (state, {
  runtimeStateManager,
  services
}) => {
  const persistedDiscoverSession = state.persistedDiscoverSession;
  if (!persistedDiscoverSession) {
    return {
      hasUnsavedChanges: false,
      unsavedTabIds: []
    };
  }
  const persistedTabIds = persistedDiscoverSession.tabs.map(tab => tab.id);
  const currentTabsIds = state.tabs.allIds;
  let tabIdsChanged = false;
  if (!(0, _lodash.isEqual)(persistedTabIds, currentTabsIds)) {
    tabIdsChanged = true;
    (0, _add_log.addLog)('[DiscoverSession] difference between initial and changed version: tab ids', {
      before: persistedTabIds,
      after: currentTabsIds
    });
  }
  const unsavedTabIds = [];
  for (const tabId of currentTabsIds) {
    const persistedTab = persistedDiscoverSession.tabs.find(tab => tab.id === tabId);
    if (!persistedTab) {
      unsavedTabIds.push(tabId);
      continue;
    }
    const tabState = (0, _tabs.selectTab)(state, tabId);
    const tabRuntimeState = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, tabId);
    const tabStateContainer = tabRuntimeState === null || tabRuntimeState === void 0 ? void 0 : tabRuntimeState.stateContainer$.getValue();
    const normalizedTab = tabStateContainer ? (0, _tab_mapping_utils.fromSavedSearchToSavedObjectTab)({
      tab: tabState,
      savedSearch: tabStateContainer.savedSearchState.getState(),
      services
    }) : (0, _tab_mapping_utils.fromTabStateToSavedObjectTab)({
      tab: tabState,
      timeRestore: Boolean(persistedTab.timeRestore),
      services
    });
    for (const stringKey of Object.keys(TAB_COMPARATORS)) {
      const key = stringKey;
      const compare = TAB_COMPARATORS[key];
      const prevField = persistedTab[key];
      const nextField = normalizedTab[key];
      if (!compare(prevField, nextField)) {
        (0, _add_log.addLog)('[DiscoverSession] difference between initial and changed version', {
          tabId,
          key,
          before: prevField,
          after: nextField
        });
        unsavedTabIds.push(tabId);
        break;
      }
    }
  }
  const hasUnsavedChanges = tabIdsChanged || unsavedTabIds.length > 0;
  if (!hasUnsavedChanges) {
    (0, _add_log.addLog)('[DiscoverSession] no difference between initial and changed version');
  }
  return {
    hasUnsavedChanges,
    unsavedTabIds
  };
};
exports.selectHasUnsavedChanges = selectHasUnsavedChanges;
const NOOP_COMPARATOR = () => true;
const defaultValueComparator = defaultValue => (a, b) => {
  // Coerce null to undefined
  a ??= undefined;
  b ??= undefined;

  // Equal if both are undefined
  if (a === undefined && b === undefined) {
    return true;
  }

  // Coerce undefined to default value
  a ??= defaultValue;
  b ??= defaultValue;
  return (0, _lodash.isEqual)(a, b);
};
const fieldComparator = (_field, defaultValue) => defaultValueComparator(defaultValue);
const FILTER_COMPARE_OPTIONS = {
  ..._esQuery.COMPARE_ALL_OPTIONS,
  state: false // We don't compare filter types (global vs appState).
};

// ad-hoc data view id can change, so we rather compare the ES|QL query itself here
const getAdjustedDataViewId = searchSource => (0, _esQuery.isOfAggregateQueryType)(searchSource.query) ? searchSource.query.esql : (0, _lodash.isObject)(searchSource.index) ? searchSource.index.id : searchSource.index;
const searchSourceComparator = (searchSourceA, searchSourceB) => {
  var _searchSourceA$filter, _searchSourceB$filter;
  const filtersA = (_searchSourceA$filter = searchSourceA.filter) !== null && _searchSourceA$filter !== void 0 ? _searchSourceA$filter : [];
  const filtersB = (_searchSourceB$filter = searchSourceB.filter) !== null && _searchSourceB$filter !== void 0 ? _searchSourceB$filter : [];
  return (
    // if a filter gets pinned and the order of filters does not change,
    // we don't show the unsaved changes badge
    (0, _discover_app_state_container.isEqualFilters)(filtersA, filtersB, FILTER_COMPARE_OPTIONS) && (0, _lodash.isEqual)(searchSourceA.query, searchSourceB.query) && getAdjustedDataViewId(searchSourceA) === getAdjustedDataViewId(searchSourceB)
  );
};

// ignore differences in title as it sometimes does not match the actual vis type/shape
const getAdjustedVisContext = visContext => (0, _unifiedHistogram.canImportVisContext)(visContext) && visContext.attributes.title ? (0, _lodash.omit)(visContext, 'attributes.title') : visContext;
const visContextComparator = (visContextA, visContextB) => {
  return (0, _lodash.isEqual)(getAdjustedVisContext(visContextA), getAdjustedVisContext(visContextB));
};
const TAB_COMPARATORS = {
  id: fieldComparator('id', ''),
  label: fieldComparator('label', ''),
  sort: fieldComparator('sort', []),
  columns: fieldComparator('columns', []),
  grid: fieldComparator('grid', {}),
  hideChart: fieldComparator('hideChart', false),
  // isTextBasedQuery is derived from the query itself and can be ignored
  isTextBasedQuery: NOOP_COMPARATOR,
  // usesAdHocDataView is derived from the data view itself and can be ignored
  usesAdHocDataView: NOOP_COMPARATOR,
  serializedSearchSource: searchSourceComparator,
  // By default, viewMode: undefined is equivalent to documents view
  // So they should be treated as same
  viewMode: fieldComparator('viewMode', _public.VIEW_MODE.DOCUMENT_LEVEL),
  hideAggregatedPreview: fieldComparator('hideAggregatedPreview', false),
  rowHeight: fieldComparator('rowHeight', 0),
  headerRowHeight: fieldComparator('headerRowHeight', 0),
  timeRestore: fieldComparator('timeRestore', false),
  timeRange: fieldComparator('timeRange', {
    from: '',
    to: ''
  }),
  refreshInterval: fieldComparator('refreshInterval', {
    pause: true,
    value: 0
  }),
  rowsPerPage: fieldComparator('rowsPerPage', 0),
  sampleSize: fieldComparator('sampleSize', 0),
  breakdownField: fieldComparator('breakdownField', ''),
  density: fieldComparator('density', _unifiedDataTable.DataGridDensity.COMPACT),
  visContext: visContextComparator,
  controlGroupJson: fieldComparator('controlGroupJson', '{}')
};