"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IncrementalIdTaskManager = exports.CasesIncrementIdTaskVersion = exports.CASES_INCREMENTAL_ID_SYNC_TASK_TYPE = exports.CASES_INCREMENTAL_ID_SYNC_TASK_ID = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _server2 = require("@kbn/task-manager-plugin/server");
var _constants = require("../../../common/constants");
var _incremental_id = require("../../services/incremental_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CASES_INCREMENTAL_ID_SYNC_TASK_TYPE = exports.CASES_INCREMENTAL_ID_SYNC_TASK_TYPE = 'cases_incremental_id_assignment';
const CASES_INCREMENTAL_ID_SYNC_TASK_ID = exports.CASES_INCREMENTAL_ID_SYNC_TASK_ID = `cases:${CASES_INCREMENTAL_ID_SYNC_TASK_TYPE}`;
const CasesIncrementIdTaskVersion = exports.CasesIncrementIdTaskVersion = '1.0.0';
class IncrementalIdTaskManager {
  constructor(taskManager, config, logger, usageCollection) {
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "internalSavedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "taskManager", void 0);
    (0, _defineProperty2.default)(this, "successErrorUsageCounter", void 0);
    this.config = config;
    this.logger = logger.get('incremental_id_task');
    this.logger.info('Registering Case Incremental ID Task Manager');
    if (usageCollection) {
      this.successErrorUsageCounter = usageCollection === null || usageCollection === void 0 ? void 0 : usageCollection.createUsageCounter('CasesIncrementalId');
    }
    taskManager.registerTaskDefinitions({
      [CASES_INCREMENTAL_ID_SYNC_TASK_TYPE]: {
        title: 'Cases Numerical ID assignment',
        description: 'Applying incremental numeric ids to cases',
        timeout: '10m',
        createTaskRunner: () => {
          if (!this.internalSavedObjectsClient) {
            throw new Error('Missing internal saved objects client.');
          }
          const casesIncrementService = new _incremental_id.CasesIncrementalIdService(this.internalSavedObjectsClient, this.logger);
          return {
            run: async () => {
              const initializedTime = new Date().toISOString();
              const startTime = performance.now();
              this.logger.debug(`Increment id task started at: ${initializedTime}`);
              casesIncrementService.startService();

              // Fetch all cases without an incremental id
              const casesWithoutIncrementalIdResponse = await casesIncrementService.getCasesWithoutIncrementalId();
              const {
                saved_objects: casesWithoutIncrementalId
              } = casesWithoutIncrementalIdResponse;
              this.logger.debug(`${casesWithoutIncrementalId.length} cases without incremental ids`);
              try {
                var _this$successErrorUsa;
                // Increment the case ids
                const processedAmount = await casesIncrementService.incrementCaseIds(casesWithoutIncrementalId);
                this.logger.debug(`Applied incremental ids to ${processedAmount} out of ${casesWithoutIncrementalId.length} cases`);
                const endTime = performance.now();
                this.logger.debug(`Task terminated ${CASES_INCREMENTAL_ID_SYNC_TASK_ID}. Task run took ${endTime - startTime}ms [ started: ${initializedTime}, ended: ${new Date().toISOString()} ]`);
                (_this$successErrorUsa = this.successErrorUsageCounter) === null || _this$successErrorUsa === void 0 ? void 0 : _this$successErrorUsa.incrementCounter({
                  counterName: 'incrementIdTaskSuccess',
                  incrementBy: 1
                });
              } catch (_) {
                var _this$successErrorUsa2;
                (_this$successErrorUsa2 = this.successErrorUsageCounter) === null || _this$successErrorUsa2 === void 0 ? void 0 : _this$successErrorUsa2.incrementCounter({
                  counterName: 'incrementIdTaskError',
                  incrementBy: 1
                });
              }
            },
            cancel: async () => {
              casesIncrementService.stopService();
              this.logger.debug(`${CASES_INCREMENTAL_ID_SYNC_TASK_ID} task run was canceled`);
            }
          };
        }
      }
    });
  }
  async setupIncrementIdTask(taskManager, core) {
    this.taskManager = taskManager;

    // Instantiate saved objects client
    const internalSavedObjectsRepository = core.savedObjects.createInternalRepository([_constants.CASE_SAVED_OBJECT, _constants.CASE_ID_INCREMENTER_SAVED_OBJECT]);
    this.internalSavedObjectsClient = new _server.SavedObjectsClient(internalSavedObjectsRepository);
    try {
      const taskDoc = await this.taskManager.get(CASES_INCREMENTAL_ID_SYNC_TASK_ID);
      const scheduledToRunInTheFuture = taskDoc.runAt.getTime() >= new Date().getTime();
      const running = taskDoc.status === _server2.TaskStatus.Claiming || taskDoc.status === _server2.TaskStatus.Running;
      if (scheduledToRunInTheFuture || running) {
        this.logger.info(`${CASES_INCREMENTAL_ID_SYNC_TASK_ID} is already ${scheduledToRunInTheFuture ? `scheduled (time: ${taskDoc.runAt})` : `running (status: ${taskDoc.status})`}. No need to schedule it again.`);
        return;
      }
    } catch (e) {
      this.logger.warn(`Could not check status of ${CASES_INCREMENTAL_ID_SYNC_TASK_ID}, will continue scheduling it.`);
    }
    this.taskManager.ensureScheduled({
      id: CASES_INCREMENTAL_ID_SYNC_TASK_ID,
      taskType: CASES_INCREMENTAL_ID_SYNC_TASK_TYPE,
      // start delayed to give the system some time to start up properly
      runAt: new Date(new Date().getTime() + this.config.taskStartDelayMinutes * 60 * 1000),
      schedule: {
        interval: `${this.config.taskIntervalMinutes}m`
      },
      params: {},
      state: {},
      scope: ['cases']
    }).then(taskInstance => {
      var _taskInstance$schedul;
      this.logger.info(`${CASES_INCREMENTAL_ID_SYNC_TASK_ID} scheduled with interval ${(_taskInstance$schedul = taskInstance.schedule) === null || _taskInstance$schedul === void 0 ? void 0 : _taskInstance$schedul.interval}`);
    }, e => {
      var _e$message;
      this.logger.error(`Error scheduling task: ${CASES_INCREMENTAL_ID_SYNC_TASK_ID}: ${e}`, (_e$message = e === null || e === void 0 ? void 0 : e.message) !== null && _e$message !== void 0 ? _e$message : e);
    });
  }
}
exports.IncrementalIdTaskManager = IncrementalIdTaskManager;