"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatESMsg = void 0;
exports.formatMsg = formatMsg;
var _lodash = _interopRequireDefault(require("lodash"));
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const has = _lodash.default.has;
const getRootCause = err => _lodash.default.get(err, 'resp.error.root_cause');

/**
 * Utilize the extended error information returned from elasticsearch
 * @param  {Error|String} err
 * @returns {string}
 */
const formatESMsg = err => {
  const rootCause = getRootCause(err);
  if (!Array.isArray(rootCause)) {
    return;
  }
  return rootCause.map(cause => cause.reason).join('\n');
};

/**
 * Formats the error message from an error object, extended elasticsearch
 * object or simple string; prepends optional second parameter to the message
 * @param  {Error|String} err
 * @param  {String} source - Prefix for message indicating source (optional)
 * @returns {string}
 */
exports.formatESMsg = formatESMsg;
function formatMsg(err, source = '') {
  let message = '';
  if (source) {
    message += source + ': ';
  }
  const esMsg = formatESMsg(err);
  if (typeof err === 'string') {
    message += err;
  } else if (esMsg) {
    message += esMsg;
  } else if (err instanceof Error) {
    message += formatMsg.describeError(err);
  } else if (has(err, 'status') && has(err, 'data')) {
    // is an Angular $http "error object"
    if (err.status === -1) {
      // status = -1 indicates that the request was failed to reach the server
      message += _i18n.i18n.translate('xpack.canvas.formatMsg.toaster.unavailableServerErrorMessage', {
        defaultMessage: 'An HTTP request has failed to connect. ' + 'Please check if the Kibana server is running and that your browser has a working connection, ' + 'or contact your system administrator.'
      });
    } else {
      message += _i18n.i18n.translate('xpack.canvas.formatMsg.toaster.errorStatusMessage', {
        defaultMessage: 'Error {errStatus} {errStatusText}: {errMessage}',
        values: {
          errStatus: err.status,
          errStatusText: err.statusText,
          errMessage: err.data.message
        }
      });
    }
  }
  return message;
}
formatMsg.describeError = function (err) {
  if (!err) return undefined;
  if (err.shortMessage) return err.shortMessage;
  if (err.body && err.body.message) return err.body.message;
  if (err.message) return err.message;
  return '' + err;
};