"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeEsQuery = executeEsQuery;
var _i18n = require("@kbn/i18n");
var _constants = require("../constants");
var _get_query_dsl = require("./get_query_dsl");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_BUCKETS_LIMIT = 65535;
async function executeEsQuery(params, esClient, shapesFilters, gteDateTime, ltDateTime) {
  const {
    entity,
    index,
    dateField,
    geoField,
    indexQuery
  } = params;
  let esFormattedQuery;
  if (indexQuery) {
    const gteEpochDateTime = gteDateTime ? new Date(gteDateTime).getTime() : null;
    const ltEpochDateTime = ltDateTime ? new Date(ltDateTime).getTime() : null;
    const dateRangeUpdatedQuery = indexQuery.language === 'kuery' ? `(${dateField} >= "${gteEpochDateTime}" and ${dateField} < "${ltEpochDateTime}") and (${indexQuery.query})` : `(${dateField}:[${gteDateTime} TO ${ltDateTime}]) AND (${indexQuery.query})`;
    esFormattedQuery = (0, _get_query_dsl.getQueryDsl)({
      query: dateRangeUpdatedQuery,
      language: indexQuery.language
    });
  }
  const esQuery = {
    index,
    size: 0,
    // do not fetch hits
    aggs: {
      shapes: {
        filters: {
          other_bucket_key: _constants.OTHER_CATEGORY,
          filters: shapesFilters
        },
        aggs: {
          entitySplit: {
            terms: {
              size: MAX_BUCKETS_LIMIT / ((Object.keys(shapesFilters).length || 1) * 2),
              field: entity
            },
            aggs: {
              entityHits: {
                top_hits: {
                  size: 1,
                  sort: [{
                    [dateField]: {
                      order: 'desc'
                    }
                  }],
                  fields: [entity, {
                    field: dateField,
                    format: 'strict_date_optional_time'
                  }, {
                    field: geoField,
                    format: 'wkt'
                  }],
                  _source: false
                }
              }
            }
          }
        }
      }
    },
    query: esFormattedQuery ? esFormattedQuery : {
      bool: {
        must: [],
        filter: [{
          match_all: {}
        }, {
          range: {
            [dateField]: {
              ...(gteDateTime ? {
                gte: gteDateTime
              } : {}),
              lt: ltDateTime,
              // 'less than' to prevent overlap between intervals
              format: 'strict_date_optional_time'
            }
          }
        }],
        should: [],
        must_not: []
      }
    },
    stored_fields: ['*'],
    docvalue_fields: [{
      field: dateField,
      format: 'date_time'
    }]
  };
  try {
    return await esClient.search(esQuery);
  } catch (err) {
    throw new Error(_i18n.i18n.translate('xpack.stackAlerts.geoContainment.entityContainmentFetchError', {
      defaultMessage: 'Unable to fetch entity containment, error: {error}',
      values: {
        error: err.message
      }
    }));
  }
}