"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.renameRiskScoreComponentTemplate = exports.MAX_PER_PAGE = void 0;
var _risk_score_data_client = require("../../risk_score/risk_score_data_client");
var _saved_object = require("../saved_object");
var _helpers = require("../../risk_score/tasks/helpers");
var _configurations = require("../../risk_score/configurations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_PER_PAGE = exports.MAX_PER_PAGE = 10_000;

/**
 * This migration renames the Risk Score component templates to include the namespace in the name. Before 8.18 all spaces used the `.risk-score-mappings` component template, we now use `.risk-score-mappings-<spacename>`.
 *
 * The migration creates the new component template and updates the index template for each space, then finally deletes the old component template.
 */
const renameRiskScoreComponentTemplate = async ({
  auditLogger,
  logger,
  getStartServices,
  kibanaVersion
}) => {
  const [coreStart] = await getStartServices();
  const soClientKibanaUser = coreStart.savedObjects.createInternalRepository();
  const esClient = coreStart.elasticsearch.client.asInternalUser;

  // Check if the legacy component templates (without the namespace in the name) exists
  const oldComponentTemplateExists = await esClient.cluster.existsComponentTemplate({
    name: _configurations.mappingComponentName
  });
  if (!oldComponentTemplateExists) {
    return;
  }

  // Get all installed Risk Engine Configurations
  const savedObjectsResponse = await soClientKibanaUser.find({
    type: _saved_object.riskEngineConfigurationTypeName,
    perPage: MAX_PER_PAGE,
    namespaces: ['*']
  });
  const settledPromises = await Promise.allSettled(savedObjectsResponse.saved_objects.map(async savedObject => {
    var _savedObject$namespac;
    const namespace = (_savedObject$namespac = savedObject.namespaces) === null || _savedObject$namespac === void 0 ? void 0 : _savedObject$namespac[0]; // We need to create one component template per space

    if (!namespace) {
      logger.error('Unexpected saved object. Risk Score saved objects must have a namespace');
      return;
    }
    logger.info(`Starting Risk Score component template migration on namespace ${namespace}`);
    const soClient = (0, _helpers.buildScopedInternalSavedObjectsClientUnsafe)({
      coreStart,
      namespace
    });
    const riskScoreDataClient = new _risk_score_data_client.RiskScoreDataClient({
      logger,
      kibanaVersion,
      esClient,
      namespace,
      soClient,
      auditLogger
    });
    await riskScoreDataClient.createOrUpdateRiskScoreComponentTemplate();
    await riskScoreDataClient.createOrUpdateRiskScoreIndexTemplate();
    logger.debug(`Risk score component template migration ran on namespace ${namespace}`);
  }));
  const rejectedPromises = settledPromises.filter(promise => promise.status === 'rejected');

  // Migration successfully ran on all spaces
  if (rejectedPromises.length === 0) {
    // Delete the legacy component template without the namespace in the name
    await esClient.cluster.deleteComponentTemplate({
      name: _configurations.mappingComponentName
    }, {
      ignore: [404]
    });
  } else {
    const errorMessages = rejectedPromises.map(promise => {
      var _promise$reason;
      return (_promise$reason = promise.reason) === null || _promise$reason === void 0 ? void 0 : _promise$reason.message;
    }).join('\n');
    throw new Error(`Risk Score component template migration failed with errors: \n${errorMessages}`);
  }
};
exports.renameRiskScoreComponentTemplate = renameRiskScoreComponentTemplate;