"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upsertPrivilegedMonitoringEntitySource = void 0;
var _std = require("@kbn/std");
var _fp = require("lodash/fp");
var _api_key = require("../auth/api_key");
var _initialisation_sources_service = require("../engine/initialisation_sources_service");
var _monitoring_entity_source = require("../saved_objects/monitoring_entity_source");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This migration runs when the privilege monitoring engine is enabled for a space.
 * This migration is required for the scenario where users installed the privilege monitoring engine in an older version.
 */
const upsertPrivilegedMonitoringEntitySource = async ({
  logger,
  getStartServices
}) => {
  const [core, {
    security,
    encryptedSavedObjects
  }] = await getStartServices();
  const soClientGlobal = core.savedObjects.createInternalRepository();
  const engineClient = new _saved_objects.PrivilegeMonitoringEngineDescriptorClient({
    soClient: soClientGlobal,
    namespace: '*' // get all engines from all namespaces
  });
  const engines = await engineClient.find();
  if (engines.total === 0) {
    logger.info('No privilege monitoring engine descriptor found. Skipping migration.');
    return;
  }

  // namespaces where engines are installed
  const namespaces = engines.saved_objects.map(savedObject => (0, _fp.first)(savedObject.namespaces));
  await (0, _std.asyncForEach)(namespaces, async namespace => {
    if (!namespace) {
      logger.error('Unexpected saved object. Monitoring Entity Source Score saved objects must have a namespace');
      return;
    }
    const apiKeyManager = (0, _api_key.getApiKeyManager)({
      core,
      logger,
      security,
      encryptedSavedObjects,
      namespace
    });
    const client = await apiKeyManager.getClient();
    if (!(client !== null && client !== void 0 && client.soClient)) {
      logger.error('[Privilege Monitoring] Unable to create Elasticsearch client from API key.');
      return undefined;
    }
    const soClient = client.soClient.asScopedToNamespace(namespace); // The saved objects client returned is unscoped
    const sourceClient = new _monitoring_entity_source.MonitoringEntitySourceDescriptorClient({
      soClient,
      namespace
    });
    const upsertSources = (0, _initialisation_sources_service.createInitialisationSourcesService)({
      descriptorClient: sourceClient,
      logger,
      auditLogger: core.security.audit.withoutRequest
    });
    await upsertSources(namespace);
  });
};
exports.upsertPrivilegedMonitoringEntitySource = upsertPrivilegedMonitoringEntitySource;