"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KNOWLEDGE_BASE_WRITE_TOOL = void 0;
var _tools = require("@langchain/core/tools");
var _zod = require("@kbn/zod");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toolDetails = {
  // note: this description is overwritten when `getTool` is called
  // local definitions exist ../elastic_assistant/server/lib/prompt/tool_prompts.ts
  // local definitions can be overwritten by security-ai-prompt integration definitions
  description: "Call this for writing details to the user's knowledge base. The knowledge base contains useful information the user wants to store between conversation contexts. Input will be the summarized knowledge base entry to store, a short UI friendly name for the entry, and whether or not the entry is required.",
  id: 'knowledge-base-write-tool',
  name: 'KnowledgeBaseWriteTool'
};
const KNOWLEDGE_BASE_WRITE_TOOL = exports.KNOWLEDGE_BASE_WRITE_TOOL = {
  ...toolDetails,
  sourceRegister: _common.APP_UI_ID,
  isSupported: params => {
    const {
      isEnabledKnowledgeBase,
      kbDataClient
    } = params;
    return isEnabledKnowledgeBase && kbDataClient != null;
  },
  async getTool(params) {
    if (!this.isSupported(params)) return null;
    const {
      telemetry,
      kbDataClient,
      logger
    } = params;
    if (kbDataClient == null) return null;
    return (0, _tools.tool)(async input => {
      logger.debug(() => `KnowledgeBaseWriteToolParams:input\n ${JSON.stringify(input, null, 2)}`);
      const knowledgeBaseEntry = {
        name: input.name,
        kbResource: 'user',
        source: 'conversation',
        required: input.required,
        text: input.query,
        type: _elasticAssistantCommon.DocumentEntryType.value
      };
      logger.debug(() => `knowledgeBaseEntry\n ${JSON.stringify(knowledgeBaseEntry, null, 2)}`);
      const resp = await kbDataClient.createKnowledgeBaseEntry({
        knowledgeBaseEntry,
        telemetry
      });
      if (resp == null) {
        return "I'm sorry, but I was unable to add this entry to your knowledge base.";
      }
      return "I've successfully saved this entry to your knowledge base. You can ask me to recall this information at any time.";
    }, {
      name: toolDetails.name,
      description: params.description || toolDetails.description,
      schema: _zod.z.object({
        name: _zod.z.string().describe(`This is what the user will use to refer to the entry in the future.`),
        query: _zod.z.string().describe(`Summary of items/things to save in the knowledge base`),
        required: _zod.z.boolean().describe(`Whether or not the entry is required to always be included in conversations. Is only true if the user explicitly asks for it to be required or always included in conversations, otherwise this is always false.`).default(false)
      }),
      tags: ['knowledge-base']
    });
  }
};