"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRules = updateRules;
exports.useUpdateRulesCache = exports.useInvalidateFindRulesQuery = exports.useFindRulesQuery = void 0;
var _reactQuery = require("@kbn/react-query");
var _react = require("react");
var _constants = require("../../../../../common/constants");
var _api = require("../api");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FIND_RULES_QUERY_KEY = ['GET', _constants.DETECTION_ENGINE_RULES_URL_FIND];
/**
 * A wrapper around useQuery provides default values to the underlying query,
 * like query key, abortion signal.
 *
 * @param queryPrefix - query prefix used to differentiate the query from other
 * findRules queries
 * @param queryArgs - fetch rules filters/pagination
 * @param queryOptions - react-query options
 * @returns useQuery result
 */
const useFindRulesQuery = (queryArgs, queryOptions) => {
  return (0, _reactQuery.useQuery)([...FIND_RULES_QUERY_KEY, queryArgs], async ({
    signal
  }) => {
    const response = await (0, _api.fetchRules)({
      signal,
      ...queryArgs
    });
    return {
      rules: response.data,
      total: response.total
    };
  }, {
    ..._constants2.DEFAULT_QUERY_OPTIONS,
    // Mark this query as immediately stale helps to avoid problems related to filtering.
    // e.g. enabled and disabled state filter require data update which happens at the backend side
    staleTime: 0,
    ...queryOptions
  });
};

/**
 * We should use this hook to invalidate the rules cache. For example, rule
 * mutations that affect rule set size, like creation or deletion, should lead
 * to cache invalidation.
 *
 * @returns A rules cache invalidation callback
 */
exports.useFindRulesQuery = useFindRulesQuery;
const useInvalidateFindRulesQuery = () => {
  const queryClient = (0, _reactQuery.useQueryClient)();
  return (0, _react.useCallback)(() => {
    /**
     * Invalidate all queries that start with FIND_RULES_QUERY_KEY. This
     * includes the in-memory query cache and paged query cache.
     */
    queryClient.invalidateQueries(FIND_RULES_QUERY_KEY, {
      refetchType: 'active'
    });
  }, [queryClient]);
};

/**
 * We should use this hook to update the rules cache when modifying rules
 * without changing the rules collection size. Use it with the new rules data
 * after operations like bulk or single rule edit or rule enabling, but not
 * when adding or removing rules. When adding/removing rules, we should
 * invalidate the cache instead.
 *
 * @returns A rules cache update callback
 */
exports.useInvalidateFindRulesQuery = useInvalidateFindRulesQuery;
const useUpdateRulesCache = () => {
  const queryClient = (0, _reactQuery.useQueryClient)();
  /**
   * Use this method to update rules data cached by react-query.
   * It is useful when we receive new rules back from a mutation query (bulk edit, etc.);
   * we can merge those rules with the existing cache to avoid an extra roundtrip to re-fetch updated rules.
   */
  return (0, _react.useCallback)(newRules => {
    queryClient.setQueriesData(FIND_RULES_QUERY_KEY, currentData => currentData ? {
      rules: updateRules(currentData.rules, newRules),
      total: currentData.total
    } : undefined);
  }, [queryClient]);
};

/**
 * Update cached rules with the new ones
 *
 * @param currentRules
 * @param newRules
 */
exports.useUpdateRulesCache = useUpdateRulesCache;
function updateRules(currentRules, newRules) {
  const newRulesMap = new Map(newRules.map(rule => [rule.id, rule]));
  if (currentRules.some(rule => newRulesMap.has(rule.id))) {
    return currentRules.map(rule => {
      var _newRulesMap$get;
      return (_newRulesMap$get = newRulesMap.get(rule.id)) !== null && _newRulesMap$get !== void 0 ? _newRulesMap$get : rule;
    });
  }
  return currentRules;
}