"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EsQueryRuleParamsSchema = void 0;
exports.validateParams = validateParams;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _utils = require("../common/utils");
var _constants = require("../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const EsQueryRuleParamsSchemaProperties = {
  size: _configSchema.schema.number({
    min: 0,
    max: _constants.ES_QUERY_MAX_HITS_PER_EXECUTION,
    meta: {
      description: 'The number of documents to pass to the configured actions when the threshold condition is met.'
    }
  }),
  timeWindowSize: _configSchema.schema.number({
    min: 1,
    meta: {
      description: 'The size of the time window (in `timeWindowUnit` units), which determines how far back to search for documents. Generally it should be a value higher than the rule check interval to avoid gaps in detection.'
    }
  }),
  excludeHitsFromPreviousRun: _configSchema.schema.boolean({
    defaultValue: true,
    meta: {
      description: 'Indicates whether to exclude matches from previous runs. If `true`, you can avoid alert duplication by excluding documents that have already been detected by the previous rule run. This option is not available when a grouping field is specified.'
    }
  }),
  timeWindowUnit: _configSchema.schema.string({
    validate: _utils.validateTimeWindowUnits,
    meta: {
      description: 'The type of units for the time window. For example: seconds, minutes, hours, or days.'
    }
  }),
  threshold: _configSchema.schema.arrayOf(_configSchema.schema.number({
    meta: {
      description: 'The threshold value that is used with the `thresholdComparator`. If the `thresholdComparator` is `between` or `notBetween`, you must specify the boundary values.'
    }
  }), {
    minSize: 1,
    maxSize: 2
  }),
  thresholdComparator: (0, _utils.getComparatorSchemaType)(_utils.validateComparator),
  aggType: _configSchema.schema.string({
    validate: _utils.validateAggType,
    defaultValue: 'count',
    meta: {
      description: 'The type of aggregation to perform.'
    }
  }),
  aggField: _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 1,
    meta: {
      description: 'The name of the numeric field that is used in the aggregation. This property is required when `aggType` is `avg`, `max`, `min` or `sum`.'
    }
  })),
  groupBy: _configSchema.schema.string({
    validate: _utils.validateGroupBy,
    defaultValue: 'all',
    meta: {
      description: 'Indicates whether the aggregation is applied over all documents (`all`), grouped by row (`row`), or split into groups (`top`) using a grouping field (`termField`) where only the top groups (up to `termSize` number of groups) are checked. If grouping is used, an alert will be created for each group when it exceeds the threshold.'
    }
  }),
  termField: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string({
    minLength: 1
  }), _configSchema.schema.arrayOf(_configSchema.schema.string(), {
    minSize: 2,
    maxSize: _constants.MAX_SELECTABLE_GROUP_BY_TERMS
  })], {
    meta: {
      description: 'The names of up to four fields that are used for grouping the aggregation. This property is required when `groupBy` is `top`.'
    }
  })),
  termSize: _configSchema.schema.maybe(_configSchema.schema.number({
    min: 1,
    meta: {
      description: 'This property is required when `groupBy` is `top`. It specifies the number of groups to check against the threshold and therefore limits the number of alerts on high cardinality fields.'
    }
  })),
  searchType: _configSchema.schema.oneOf([_configSchema.schema.literal('searchSource'), _configSchema.schema.literal('esQuery'), _configSchema.schema.literal('esqlQuery')], {
    meta: {
      description: 'The type of query For example: `esQuery` for Elasticsearch Query DSL or `esqlQuery` for Elasticsearch Query Language (ES|QL).'
    },
    defaultValue: 'esQuery'
  }),
  timeField: _configSchema.schema.conditional(_configSchema.schema.siblingRef('searchType'), _configSchema.schema.literal('esQuery'), _configSchema.schema.string({
    minLength: 1
  }), _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 1
  })), {
    meta: {
      description: 'The field that is used to calculate the time window.'
    }
  }),
  searchConfiguration: _configSchema.schema.conditional(_configSchema.schema.siblingRef('searchType'), _configSchema.schema.literal('searchSource'), _configSchema.schema.object({}, {
    unknowns: 'allow'
  }), _configSchema.schema.never(), {
    meta: {
      description: 'The query definition, which uses KQL or Lucene to fetch the documents from Elasticsearch.'
    }
  }),
  // esQuery rule params only
  esQuery: _configSchema.schema.conditional(_configSchema.schema.siblingRef('searchType'), _configSchema.schema.literal('esQuery'), _configSchema.schema.string({
    minLength: 1
  }), _configSchema.schema.never()),
  index: _configSchema.schema.conditional(_configSchema.schema.siblingRef('searchType'), _configSchema.schema.literal('esQuery'), _configSchema.schema.arrayOf(_configSchema.schema.string({
    minLength: 1
  }), {
    minSize: 1
  }), _configSchema.schema.never(), {
    meta: {
      description: 'The indices to query.'
    }
  }),
  esqlQuery: _configSchema.schema.conditional(_configSchema.schema.siblingRef('searchType'), _configSchema.schema.literal('esqlQuery'), _configSchema.schema.object({
    esql: _configSchema.schema.string({
      minLength: 1
    })
  }), _configSchema.schema.never(), {
    meta: {
      description: 'The query definition in Elasticsearch Query Language.'
    }
  }),
  sourceFields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
    label: _configSchema.schema.string(),
    searchPath: _configSchema.schema.string()
  }), {
    maxSize: _constants.MAX_SELECTABLE_SOURCE_FIELDS,
    meta: {
      description: 'The sourceFields param is ignored.'
    }
  }))
};

// rule type parameters

function isSearchSourceRule(searchType) {
  return searchType === 'searchSource';
}
function isEsqlQueryRule(searchType) {
  return searchType === 'esqlQuery';
}

// using direct type not allowed, circular reference, so body is typed to any
function validateParams(anyParams) {
  const {
    esQuery,
    thresholdComparator,
    threshold,
    searchType,
    aggType,
    aggField,
    groupBy,
    termField,
    termSize
  } = anyParams;
  if (isEsqlQueryRule(searchType)) {
    const {
      timeField
    } = anyParams;
    if (!timeField) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.esQuery.esqlTimeFieldErrorMessage', {
        defaultMessage: '[timeField]: is required'
      });
    }
    if (thresholdComparator !== _constants.Comparator.GT) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.esQuery.esqlThresholdComparatorErrorMessage', {
        defaultMessage: '[thresholdComparator]: is required to be greater than'
      });
    }
    if (threshold && threshold[0] !== 0) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.esQuery.esqlThresholdErrorMessage', {
        defaultMessage: '[threshold]: is required to be 0'
      });
    }

    // The esqlQuery type does not validate groupBy, as any groupBy other than 'row' is considered to be 'all'
    return;
  }
  if (_utils.betweenComparators.has(thresholdComparator) && threshold.length === 1) {
    return _i18n.i18n.translate('responseOps.ruleParams.esQuery.invalidThreshold2ErrorMessage', {
      defaultMessage: '[threshold]: must have two elements for the "{thresholdComparator}" comparator',
      values: {
        thresholdComparator
      }
    });
  }
  if (aggType !== 'count' && !aggField) {
    return _i18n.i18n.translate('responseOps.ruleParams.esQuery.aggTypeRequiredErrorMessage', {
      defaultMessage: '[aggField]: must have a value when [aggType] is "{aggType}"',
      values: {
        aggType
      }
    });
  }

  // check grouping
  if (groupBy === 'top') {
    if (termField == null) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.esQuery.termFieldRequiredErrorMessage', {
        defaultMessage: '[termField]: termField required when [groupBy] is top'
      });
    }
    if (termSize == null) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.esQuery.termSizeRequiredErrorMessage', {
        defaultMessage: '[termSize]: termSize required when [groupBy] is top'
      });
    }
    if (termSize > _constants.MAX_GROUPS) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.esQuery.invalidTermSizeMaximumErrorMessage', {
        defaultMessage: '[termSize]: must be less than or equal to {maxGroups}',
        values: {
          maxGroups: _constants.MAX_GROUPS
        }
      });
    }
  }
  if (groupBy === 'row') {
    return _i18n.i18n.translate('xpack.responseOps.ruleParams.esQuery.invalidRowGroupByErrorMessage', {
      defaultMessage: '[groupBy]: groupBy should be all or top when [searchType] is not esqlQuery'
    });
  }
  if (isSearchSourceRule(searchType)) {
    return;
  }
  try {
    const parsedQuery = JSON.parse(esQuery);
    if (parsedQuery && !parsedQuery.query) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.esQuery.missingEsQueryErrorMessage', {
        defaultMessage: '[esQuery]: must contain "query"'
      });
    }
  } catch (err) {
    return _i18n.i18n.translate('xpack.responseOps.ruleParams.esQuery.invalidEsQueryErrorMessage', {
      defaultMessage: '[esQuery]: must be valid JSON'
    });
  }
}
const EsQueryRuleParamsSchema = exports.EsQueryRuleParamsSchema = _configSchema.schema.object(EsQueryRuleParamsSchemaProperties, {
  validate: validateParams
});