"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateBaseComparisonDiff = calculateBaseComparisonDiff;
exports.convertRowToFrame = convertRowToFrame;
exports.getColorLabel = getColorLabel;
exports.getFunctionsRows = getFunctionsRows;
exports.getTotalCount = void 0;
exports.scaleAndRoundValue = scaleAndRoundValue;
var _lodash = require("lodash");
var _profilingUtils = require("@kbn/profiling-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getColorLabel(percent) {
  if (percent === 0) {
    return {
      color: 'text',
      label: `0%`,
      icon: undefined
    };
  }
  const color = percent < 0 ? 'success' : 'danger';
  const icon = percent < 0 ? 'sortUp' : 'sortDown';
  const isSmallPercent = Math.abs(percent) <= 0.01;
  const value = isSmallPercent ? '<0.01' : Math.abs(percent).toFixed(2);
  if (isFinite(percent)) {
    return {
      color,
      label: `${value}%`,
      icon
    };
  }
  return {
    color: 'text',
    label: undefined,
    icon: undefined
  };
}
function scaleValue({
  value,
  scaleFactor = 1
}) {
  return value * scaleFactor;
}
function scaleAndRoundValue({
  value,
  scaleFactor = 1
}) {
  return Math.round(scaleValue({
    value,
    scaleFactor
  }));
}
const getTotalCount = topNFunctions => {
  var _topNFunctions$selfCP;
  return (_topNFunctions$selfCP = topNFunctions === null || topNFunctions === void 0 ? void 0 : topNFunctions.selfCPU) !== null && _topNFunctions$selfCP !== void 0 ? _topNFunctions$selfCP : 0;
};
exports.getTotalCount = getTotalCount;
function getFunctionsRows({
  baselineScaleFactor,
  comparisonScaleFactor,
  comparisonTopNFunctions,
  topNFunctions,
  totalSeconds,
  calculateImpactEstimates,
  functionNameSearchQuery
}) {
  if (!topNFunctions || !topNFunctions.TotalCount || topNFunctions.TotalCount === 0) {
    return [];
  }
  const comparisonDataById = comparisonTopNFunctions ? (0, _lodash.keyBy)(comparisonTopNFunctions.TopN, 'Id') : {};
  return topNFunctions.TopN.filter(topN => topN.CountExclusive >= 0).filter(topN => {
    if (functionNameSearchQuery) {
      const functionName = (0, _profilingUtils.getCalleeFunction)(topN.Frame);
      return functionName.toLowerCase().includes(functionNameSearchQuery.toLowerCase());
    }
    return true;
  }).map((topN, i) => {
    const comparisonRow = comparisonDataById === null || comparisonDataById === void 0 ? void 0 : comparisonDataById[topN.Id];
    const scaledSelfCPU = scaleValue({
      value: topN.CountExclusive,
      scaleFactor: baselineScaleFactor
    });
    const totalCPUPerc = topN.CountInclusive / topNFunctions.selfCPU * 100;
    const selfCPUPerc = topN.CountExclusive / topNFunctions.selfCPU * 100;
    const impactEstimates = totalSeconds > 0 ? calculateImpactEstimates({
      countExclusive: topN.CountExclusive,
      countInclusive: topN.CountInclusive,
      totalSamples: topNFunctions.selfCPU,
      totalSeconds
    }) : undefined;
    function calculateDiff() {
      if (comparisonTopNFunctions && comparisonRow) {
        const comparisonScaledSelfCPU = scaleValue({
          value: comparisonRow.CountExclusive,
          scaleFactor: comparisonScaleFactor
        });
        const scaledDiffSamples = scaledSelfCPU - comparisonScaledSelfCPU;
        return {
          id: comparisonRow.Id,
          rank: topN.Rank - comparisonRow.Rank,
          samples: scaledDiffSamples,
          selfCPU: comparisonRow.CountExclusive,
          totalCPU: comparisonRow.CountInclusive,
          selfCPUPerc: selfCPUPerc - comparisonRow.CountExclusive / comparisonTopNFunctions.selfCPU * 100,
          totalCPUPerc: totalCPUPerc - comparisonRow.CountInclusive / comparisonTopNFunctions.selfCPU * 100,
          selfAnnualCO2kgs: comparisonRow.selfAnnualCO2kgs,
          selfAnnualCostUSD: comparisonRow.selfAnnualCostUSD,
          totalAnnualCO2kgs: comparisonRow.totalAnnualCO2kgs,
          totalAnnualCostUSD: comparisonRow.totalAnnualCostUSD
        };
      }
    }
    return {
      id: topN.Id,
      rank: topN.Rank,
      frame: topN.Frame,
      samples: scaledSelfCPU,
      selfCPUPerc,
      totalCPUPerc,
      selfCPU: topN.CountExclusive,
      totalCPU: topN.CountInclusive,
      impactEstimates,
      selfAnnualCO2kgs: topN.selfAnnualCO2kgs,
      selfAnnualCostUSD: topN.selfAnnualCostUSD,
      totalAnnualCO2kgs: topN.totalAnnualCO2kgs,
      totalAnnualCostUSD: topN.totalAnnualCostUSD,
      subGroups: topN.subGroups,
      diff: calculateDiff()
    };
  });
}
function calculateBaseComparisonDiff({
  baselineValue,
  baselineScaleFactor,
  comparisonValue,
  comparisonScaleFactor,
  formatValue
}) {
  const scaledBaselineValue = scaleValue({
    value: baselineValue,
    scaleFactor: baselineScaleFactor
  });
  const baseValue = formatValue ? formatValue(scaledBaselineValue) : scaledBaselineValue.toLocaleString();
  if (comparisonValue === 0) {
    return {
      baseValue
    };
  }
  const scaledComparisonValue = scaleValue({
    value: comparisonValue,
    scaleFactor: comparisonScaleFactor
  });
  const diffSamples = scaledComparisonValue - scaledBaselineValue;
  const percentDiffDelta = diffSamples / (scaledComparisonValue - diffSamples) * 100;
  const {
    color,
    icon,
    label
  } = getColorLabel(percentDiffDelta);
  return {
    baseValue,
    comparisonValue: formatValue ? formatValue(scaledComparisonValue) : scaledComparisonValue.toLocaleString(),
    percentDiffDelta,
    color,
    icon,
    label
  };
}
function convertRowToFrame(row) {
  return {
    addressOrLine: row.frame.AddressOrLine,
    countExclusive: row.selfCPU,
    countInclusive: row.totalCPU,
    exeFileName: row.frame.ExeFileName,
    fileID: row.frame.FileID,
    frameType: row.frame.FrameType,
    functionName: row.frame.FunctionName,
    sourceFileName: row.frame.SourceFilename,
    sourceLine: row.frame.SourceLine,
    selfAnnualCO2Kgs: row.selfAnnualCO2kgs,
    totalAnnualCO2Kgs: row.totalAnnualCO2kgs,
    selfAnnualCostUSD: row.selfAnnualCostUSD,
    totalAnnualCostUSD: row.totalAnnualCostUSD,
    subGroups: row.subGroups
  };
}