"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSendMessageMutation = void 0;
var _reactQuery = require("@kbn/react-query");
var _react = require("react");
var _use_conversation = require("../../hooks/use_conversation");
var _use_conversation_actions = require("../../hooks/use_conversation_actions");
var _use_conversation_id = require("../../hooks/use_conversation_id");
var _use_onechat_service = require("../../hooks/use_onechat_service");
var _use_report_error = require("../../hooks/use_report_error");
var _mutation_keys = require("../../mutation_keys");
var _use_pending_message_state = require("./use_pending_message_state");
var _use_subscribe_to_chat_events = require("./use_subscribe_to_chat_events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useSendMessageMutation = ({
  connectorId
} = {}) => {
  const {
    chatService
  } = (0, _use_onechat_service.useOnechatServices)();
  const {
    reportConverseError
  } = (0, _use_report_error.useReportConverseError)();
  const conversationActions = (0, _use_conversation_actions.useConversationActions)();
  const [isResponseLoading, setIsResponseLoading] = (0, _react.useState)(false);
  const [agentReasoning, setAgentReasoning] = (0, _react.useState)(null);
  const conversationId = (0, _use_conversation_id.useConversationId)();
  const isMutatingNewConversationRef = (0, _react.useRef)(false);
  const agentId = (0, _use_conversation.useAgentId)();
  const messageControllerRef = (0, _react.useRef)(null);
  const {
    pendingMessageState: {
      error,
      pendingMessage
    },
    setPendingMessage,
    removePendingMessage,
    setError,
    removeError
  } = (0, _use_pending_message_state.usePendingMessageState)({
    conversationId
  });
  const subscribeToChatEvents = (0, _use_subscribe_to_chat_events.useSubscribeToChatEvents)({
    setAgentReasoning,
    setIsResponseLoading,
    isAborted: () => {
      var _messageControllerRef, _messageControllerRef2;
      return Boolean(messageControllerRef === null || messageControllerRef === void 0 ? void 0 : (_messageControllerRef = messageControllerRef.current) === null || _messageControllerRef === void 0 ? void 0 : (_messageControllerRef2 = _messageControllerRef.signal) === null || _messageControllerRef2 === void 0 ? void 0 : _messageControllerRef2.aborted);
    }
  });
  const sendMessage = ({
    message
  }) => {
    var _messageControllerRef3;
    const signal = (_messageControllerRef3 = messageControllerRef.current) === null || _messageControllerRef3 === void 0 ? void 0 : _messageControllerRef3.signal;
    if (!signal) {
      return Promise.reject(new Error('Abort signal not present'));
    }
    const events$ = chatService.chat({
      signal,
      input: message,
      conversationId,
      agentId,
      connectorId
    });
    return subscribeToChatEvents(events$);
  };
  const {
    mutate,
    isLoading
  } = (0, _reactQuery.useMutation)({
    mutationKey: _mutation_keys.mutationKeys.sendMessage,
    mutationFn: sendMessage,
    onMutate: ({
      message
    }) => {
      const isNewConversation = !conversationId;
      isMutatingNewConversationRef.current = isNewConversation;
      setPendingMessage(message);
      removeError();
      messageControllerRef.current = new AbortController();
      conversationActions.addOptimisticRound({
        userMessage: message
      });
      if (isNewConversation) {
        if (!agentId) {
          throw new Error('Agent id must be defined for a new conversation');
        }
        conversationActions.setAgentId(agentId);
      }
      setIsResponseLoading(true);
    },
    onSettled: () => {
      conversationActions.invalidateConversation();
      messageControllerRef.current = null;
      setAgentReasoning(null);
    },
    onSuccess: () => {
      removePendingMessage();
      if (isMutatingNewConversationRef.current) {
        conversationActions.removeNewConversationQuery();
      }
    },
    onError: err => {
      setIsResponseLoading(false);
      reportConverseError(err, {
        connectorId
      });
      setError(err);
      // When we error, we should immediately remove the round rather than waiting for a refetch after invalidation
      // Otherwise, the error round and the optimistic round will be visible together.
      conversationActions.removeOptimisticRound();
    }
  });
  const canCancel = isLoading;
  const cancel = () => {
    var _messageControllerRef4;
    if (!canCancel) {
      return;
    }
    removePendingMessage();
    (_messageControllerRef4 = messageControllerRef.current) === null || _messageControllerRef4 === void 0 ? void 0 : _messageControllerRef4.abort();
  };
  return {
    sendMessage: mutate,
    isResponseLoading,
    error,
    pendingMessage,
    agentReasoning,
    retry: () => {
      if (
      // Retrying should not be allowed if a response is still being fetched
      // or if we're not in an error state
      isResponseLoading || !error) {
        return;
      }
      if (!pendingMessage) {
        // Should never happen
        // If we are in an error state, pending message will be present
        throw new Error('Pending message is not present');
      }
      mutate({
        message: pendingMessage
      });
    },
    canCancel,
    cancel,
    // Cleaning only makes sense in the context of an error state on a new conversation round
    // The user can click "New" to clear the pending round and error
    cleanConversation: () => {
      conversationActions.removeOptimisticRound();
      removeError();
      removePendingMessage();
    }
  };
};
exports.useSendMessageMutation = useSendMessageMutation;