"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLogstashMonitoringMethods = getLogstashMonitoringMethods;
exports.getLogstashStats = getLogstashStats;
exports.logstashMonitoringInstances = void 0;
var _constants = require("../../common/constants");
var _logstash_self_monitoring = require("./logstash_self_monitoring");
var _logstash_metricbeat_monitoring = require("./logstash_metricbeat_monitoring");
var _logstash_agent_monitoring = require("./logstash_agent_monitoring");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SELF_MONITORING = 'self';
const METRICBEAT_MONITORING = 'metricbeat';
const AGENT_MONITORING = 'agent';
const logstashMonitoringInstances = exports.logstashMonitoringInstances = {
  self: new _logstash_self_monitoring.LogstashSelfMonitoring(),
  metricbeat: new _logstash_metricbeat_monitoring.LogstashMetricbeatMonitoring(),
  agent: new _logstash_agent_monitoring.LogstashAgentMonitoring()
};

/*
 * Call the function for fetching and summarizing Logstash stats
 * @return {Object} - Logstash stats in an object keyed by the cluster UUIDs
 */
async function getLogstashStats(callCluster, clusterUuids, start, end) {
  const options = {
    clusters: {},
    // the result object to be built up
    allEphemeralIds: {},
    allHostIds: {},
    versions: {},
    plugins: {}
  };
  const monitoringClusterInfo = await callCluster.info();
  const monitoringClusterUuid = monitoringClusterInfo.cluster_uuid;

  // figure out the monitoring methods cluster is using based on the Logstash metrics indices
  // mostly single method will be resolved
  // multiple monitoring methods case might be due to migration (ex: from self to metricbeat)
  const monitoringMethods = await getLogstashMonitoringMethods(callCluster);

  // collect all _method_ (:self, :metricbeat, :agent) metrics in a given period
  for (const monitoringMethod of monitoringMethods) {
    const monitoringInstance = logstashMonitoringInstances[monitoringMethod];
    if (monitoringInstance) {
      await monitoringInstance.collectMetrics(callCluster, clusterUuids, monitoringClusterUuid, start, end, options);
    }
  }
  return options.clusters;
}
async function getLogstashMonitoringMethods(callCluster) {
  const response = await callCluster.cat.indices({
    index: _constants.INDEX_PATTERN_LOGSTASH,
    format: 'json'
  }, {
    headers: {
      'X-QUERY-SOURCE': _constants.TELEMETRY_QUERY_SOURCE
    }
  });
  const monitoringMethods = [];
  for (const record of response) {
    if (record.index.indexOf('monitoring-logstash-') !== -1) {
      if (record.index.indexOf(_constants.METRICBEAT_INDEX_NAME_UNIQUE_TOKEN) !== -1) {
        // legacy driven metricbeat monitoring
        if (!monitoringMethods.includes(METRICBEAT_MONITORING)) {
          monitoringMethods.push(METRICBEAT_MONITORING);
          logstashMonitoringInstances.metricbeat.setIndexPattern('legacy');
        }
      } else {
        if (!monitoringMethods.includes(SELF_MONITORING)) {
          monitoringMethods.push(SELF_MONITORING);
        }
      }
    } else if (record.index.indexOf('metrics-logstash.node') !== -1) {
      if (!monitoringMethods.includes(AGENT_MONITORING)) {
        monitoringMethods.push(AGENT_MONITORING);
      }
    } else if (record.index.indexOf('metrics-logstash.stack_monitoring') !== -1) {
      if (!monitoringMethods.includes(METRICBEAT_MONITORING)) {
        monitoringMethods.push(METRICBEAT_MONITORING);
        logstashMonitoringInstances.metricbeat.setIndexPattern('stack');
      }
    }
  }
  return monitoringMethods;
}