"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useIndexData = void 0;
var _react = require("react");
var _useMountedState = _interopRequireDefault(require("react-use/lib/useMountedState"));
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _mlRuntimeFieldUtils = require("@kbn/ml-runtime-field-utils");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _mlDataGrid = require("@kbn/ml-data-grid");
var _kibana = require("../../../../contexts/kibana");
var _data_loader = require("../../../../datavisualizer/index_based/data_loader");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getRuntimeFieldColumns(runtimeMappings) {
  return Object.keys(runtimeMappings).map(id => {
    let field = runtimeMappings[id];
    if (Array.isArray(field)) {
      field = field[0];
    }
    const schema = (0, _mlDataGrid.getDataGridSchemaFromESFieldType)(field.type);
    return {
      id,
      schema,
      isExpandable: schema !== 'boolean',
      isRuntimeFieldColumn: true
    };
  });
}
function getDataViewColumns(dataView, fieldsFilter) {
  const {
    fields
  } = dataView;
  return fields.filter(field => fieldsFilter.includes(field.name)).map(field => {
    const schema =
    // @ts-expect-error field is not DataViewField
    (0, _mlDataGrid.getDataGridSchemaFromESFieldType)(field.type) || (0, _mlDataGrid.getDataGridSchemaFromKibanaFieldType)(field);
    return {
      id: field.name,
      schema,
      isExpandable: schema !== 'boolean',
      isRuntimeFieldColumn: false
    };
  });
}
const useIndexData = (dataView, query, toastNotifications, runtimeMappings) => {
  const isMounted = (0, _useMountedState.default)();
  const {
    services: {
      data: {
        dataViews: dataViewsService
      }
    }
  } = (0, _kibana.useMlKibana)();
  const mlApi = (0, _kibana.useMlApi)();
  const [dataViewFields, setDataViewFields] = (0, _react.useState)();
  const [timeRangeMs, setTimeRangeMs] = (0, _react.useState)();
  const abortController = (0, _react.useRef)(new AbortController());

  // To be used for data grid column selection
  // and will be applied to doc and chart queries.
  const combinedRuntimeMappings = (0, _react.useMemo)(() => (0, _mlRuntimeFieldUtils.getCombinedRuntimeMappings)(dataView, runtimeMappings), [dataView, runtimeMappings]);
  (0, _react.useEffect)(() => {
    async function fetchPopulatedFields() {
      if (abortController.current) {
        abortController.current.abort();
        abortController.current = new AbortController();
      }
      setErrorMessage('');
      setStatus(_mlDataGrid.INDEX_STATUS.LOADING);
      try {
        const nonEmptyFields = await dataViewsService.getFieldsForIndexPattern(dataView, {
          includeEmptyFields: false,
          // dummy filter, if no filter was provided the function would return all fields.
          indexFilter: {
            bool: {
              must: {
                match_all: {}
              }
            }
          },
          runtimeMappings: combinedRuntimeMappings,
          abortSignal: abortController.current.signal
        });
        const populatedFields = nonEmptyFields.map(field => field.name);
        if (isMounted()) {
          setDataViewFields((0, _mlDataGrid.getPopulatedFieldsFromKibanaDataView)(dataView, populatedFields));
        }
      } catch (e) {
        if ((e === null || e === void 0 ? void 0 : e.name) !== 'AbortError') {
          setErrorMessage((0, _mlErrorUtils.extractErrorMessage)(e));
          setStatus(_mlDataGrid.INDEX_STATUS.ERROR);
        }
      }
    }
    fetchPopulatedFields();
    return () => {
      abortController.current.abort();
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // Available data grid columns, will be a combination of index pattern and runtime fields.
  const [columns, setColumns] = (0, _react.useState)([]);
  (0, _react.useEffect)(() => {
    if (Array.isArray(dataViewFields)) {
      const cols = [...getDataViewColumns(dataView, dataViewFields), ...(combinedRuntimeMappings ? getRuntimeFieldColumns(combinedRuntimeMappings) : [])];

      // de-duplicate columns by id to avoid runtime fields being added twice
      const uniqueCols = Array.from(new Map(cols.map(item => [item.id, item])).values());
      setColumns(uniqueCols);
    }
  }, [dataView, dataViewFields, combinedRuntimeMappings]);
  const dataGrid = (0, _mlDataGrid.useDataGrid)(columns);
  const {
    pagination,
    resetPagination,
    setErrorMessage,
    setRowCountInfo,
    setStatus,
    setTableItems,
    sortingColumns,
    tableItems
  } = dataGrid;
  (0, _react.useEffect)(() => {
    resetPagination();
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [JSON.stringify(query)]);
  (0, _react.useEffect)(() => {
    async function fetchIndexData() {
      var _dataView$getTimeFiel;
      setErrorMessage('');
      setStatus(_mlDataGrid.INDEX_STATUS.LOADING);
      const timeFieldName = (_dataView$getTimeFiel = dataView.getTimeField()) === null || _dataView$getTimeFiel === void 0 ? void 0 : _dataView$getTimeFiel.name;
      const sort = sortingColumns.reduce((s, column) => {
        s[column.id] = {
          order: column.direction
        };
        return s;
      }, {});
      const esSearchRequest = {
        index: dataView.title,
        body: {
          query,
          from: pagination.pageIndex * pagination.pageSize,
          size: pagination.pageSize,
          fields: [...(dataViewFields !== null && dataViewFields !== void 0 ? dataViewFields : []), ...((0, _mlRuntimeFieldUtils.isRuntimeMappings)(combinedRuntimeMappings) ? Object.keys(combinedRuntimeMappings) : [])],
          _source: false,
          ...(Object.keys(sort).length > 0 ? {
            sort
          } : {}),
          ...((0, _mlRuntimeFieldUtils.isRuntimeMappings)(combinedRuntimeMappings) ? {
            runtime_mappings: combinedRuntimeMappings
          } : {}),
          ...(timeFieldName ? {
            aggs: {
              earliest: {
                min: {
                  field: timeFieldName
                }
              },
              latest: {
                max: {
                  field: timeFieldName
                }
              }
            }
          } : {})
        }
      };
      try {
        const resp = await mlApi.esSearch(esSearchRequest);
        if (resp.aggregations && (0, _mlIsPopulatedObject.isPopulatedObject)(resp.aggregations.earliest, ['value']) && (0, _mlIsPopulatedObject.isPopulatedObject)(resp.aggregations.latest, ['value'])) {
          setTimeRangeMs({
            from: resp.aggregations.earliest.value,
            to: resp.aggregations.latest.value
          });
        }
        const docs = resp.hits.hits.map(d => {
          var _d$fields;
          return (0, _mlDataGrid.getProcessedFields)((_d$fields = d.fields) !== null && _d$fields !== void 0 ? _d$fields : {});
        });
        setRowCountInfo({
          rowCount: typeof resp.hits.total === 'number' ? resp.hits.total : resp.hits.total.value,
          rowCountRelation: typeof resp.hits.total === 'number' ? 'eq' : resp.hits.total.relation
        });
        setTableItems(docs);
        setStatus(_mlDataGrid.INDEX_STATUS.LOADED);
      } catch (e) {
        setErrorMessage((0, _mlErrorUtils.extractErrorMessage)(e));
        setStatus(_mlDataGrid.INDEX_STATUS.ERROR);
      }
    }
    if (dataViewFields !== undefined && query !== undefined) {
      fetchIndexData();
    }
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [dataView.title, dataViewFields,
  // eslint-disable-next-line react-hooks/exhaustive-deps
  JSON.stringify([query, pagination, sortingColumns, combinedRuntimeMappings])]);
  const dataLoader = (0, _react.useMemo)(() => new _data_loader.DataLoader(dataView, mlApi),
  // eslint-disable-next-line react-hooks/exhaustive-deps
  [dataView]);
  (0, _react.useEffect)(() => {
    async function fetchColumnChartsData(fieldHistogramsQuery) {
      try {
        const columnChartsData = await dataLoader.loadFieldHistograms(columns.filter(cT => dataGrid.visibleColumns.includes(cT.id)).map(cT => ({
          fieldName: cT.id,
          type: (0, _mlDataGrid.getFieldType)(cT.schema)
        })), fieldHistogramsQuery, _mlAggUtils.DEFAULT_SAMPLER_SHARD_SIZE, combinedRuntimeMappings);
        dataGrid.setColumnCharts(columnChartsData);
      } catch (e) {
        (0, _mlDataGrid.showDataGridColumnChartErrorMessageToast)(e, toastNotifications);
      }
    }
    if (dataGrid.chartsVisible && query !== undefined) {
      fetchColumnChartsData(query);
    }
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [dataGrid.chartsVisible, dataView.title,
  // eslint-disable-next-line react-hooks/exhaustive-deps
  JSON.stringify([query, dataGrid.visibleColumns, runtimeMappings])]);
  const renderCellValue = (0, _mlDataGrid.useRenderCellValue)(dataView, pagination, tableItems);
  return {
    ...dataGrid,
    dataViewFields,
    renderCellValue,
    timeRangeMs
  };
};
exports.useIndexData = useIndexData;