"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LensClient = void 0;
var _lodash = require("lodash");
var _constants = require("../../common/constants");
var _transforms = require("../../common/transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This type is to allow `visualizationType` to be `null` in the public context.
 *
 * The stored attributes must have a `visualizationType`.
 */

class LensClient {
  constructor(http) {
    this.http = http;
  }
  async get(id) {
    const {
      data,
      meta
    } = await this.http.get(`${_constants.LENS_VIS_API_PATH}/${id}`, {
      version: _constants.LENS_API_VERSION
    });
    return {
      item: _transforms.ConfigBuilderStub.in(data),
      meta // TODO: see if we still need this meta data
    };
  }
  async create({
    description,
    visualizationType,
    state,
    title,
    version
  }, references, options = {}) {
    if (visualizationType === null) {
      throw new Error('Missing visualization type');
    }
    const body = {
      // TODO: Find a better way to conditionally omit id
      data: (0, _lodash.omit)(_transforms.ConfigBuilderStub.out({
        id: '',
        description,
        visualizationType,
        state,
        title,
        version,
        references
      }), 'id'),
      options
    };
    const {
      data,
      meta
    } = await this.http.post(_constants.LENS_VIS_API_PATH, {
      body: JSON.stringify(body),
      version: _constants.LENS_API_VERSION
    });
    return {
      item: _transforms.ConfigBuilderStub.in(data),
      meta
    };
  }
  async update(id, {
    description,
    visualizationType,
    state,
    title,
    version
  }, references, options = {}) {
    if (visualizationType === null) {
      throw new Error('Missing visualization type');
    }
    const body = {
      // TODO: Find a better way to conditionally omit id
      data: (0, _lodash.omit)(_transforms.ConfigBuilderStub.out({
        id: '',
        description,
        visualizationType,
        state,
        title,
        version,
        references
      }), 'id'),
      options
    };
    const {
      data,
      meta
    } = await this.http.put(`${_constants.LENS_VIS_API_PATH}/${id}`, {
      body: JSON.stringify(body),
      version: _constants.LENS_API_VERSION
    });
    return {
      item: _transforms.ConfigBuilderStub.in(data),
      meta
    };
  }
  async delete(id) {
    var _response$response$ok, _response$response;
    const response = await this.http.delete(`${_constants.LENS_VIS_API_PATH}/${id}`, {
      asResponse: true,
      version: _constants.LENS_API_VERSION
    });
    const success = (_response$response$ok = (_response$response = response.response) === null || _response$response === void 0 ? void 0 : _response$response.ok) !== null && _response$response$ok !== void 0 ? _response$response$ok : false;
    return {
      success
    }; // TODO remove if not used
  }
  async search({
    query,
    page,
    perPage,
    fields,
    searchFields
  }) {
    // TODO add all CM search options to query
    const result = await this.http.get(_constants.LENS_VIS_API_PATH, {
      query: {
        query,
        page,
        perPage,
        fields,
        searchFields
      },
      version: _constants.LENS_API_VERSION
    });
    return result.data.map(({
      data
    }) => ({
      ...data,
      attributes: _transforms.ConfigBuilderStub.in(data)
    }));
  }
}
exports.LensClient = LensClient;