"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkIntegrationsAllPrivilegesForSpaces = checkIntegrationsAllPrivilegesForSpaces;
exports.installRuleAssetsHandler = exports.installPackageKibanaAssetsHandler = exports.deletePackageKibanaAssetsHandler = void 0;
var _errors = require("../../errors");
var _services = require("../../services");
var _install = require("../../services/epm/kibana/assets/install");
var _get = require("../../services/epm/packages/get");
var _archive_iterator = require("../../services/epm/archive/archive_iterator");
var _step_create_alerting_rules = require("../../services/epm/packages/install_state_machine/steps/step_create_alerting_rules");
var _http_authorization_header = require("../../../common/http_authorization_header");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function checkIntegrationsAllPrivilegesForSpaces(request, spaceIds) {
  const security = _services.appContextService.getSecurity();
  const res = await security.authz.checkPrivilegesWithRequest(request).atSpaces(spaceIds, {
    kibana: [security.authz.actions.api.get(`integrations-all`)]
  });
  if (!res.hasAllRequested) {
    throw new _errors.FleetError(`No enough permissions to install assets in spaces ${spaceIds.join(', ')}`);
  }
}
const installPackageKibanaAssetsHandler = async (context, request, response) => {
  var _request$body, _request$body$space_i, _request$body3;
  const fleetContext = await context.fleet;
  const savedObjectsClient = fleetContext.internalSoClient;
  const logger = _services.appContextService.getLogger();
  const spaceId = fleetContext.spaceId;
  const {
    pkgName,
    pkgVersion
  } = request.params;
  if ((_request$body = request.body) !== null && _request$body !== void 0 && _request$body.space_ids) {
    var _request$body2;
    await checkIntegrationsAllPrivilegesForSpaces(request, (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.space_ids);
  }
  const installedPkgWithAssets = await (0, _get.getInstalledPackageWithAssets)({
    savedObjectsClient,
    pkgName,
    logger
  });
  const installation = await (0, _get.getInstallationObject)({
    pkgName,
    savedObjectsClient
  });
  if (!installation || !installedPkgWithAssets || (installedPkgWithAssets === null || installedPkgWithAssets === void 0 ? void 0 : installedPkgWithAssets.installation.version) !== pkgVersion) {
    throw new _errors.FleetNotFoundError('Requested version is not installed');
  }
  const {
    packageInfo
  } = installedPkgWithAssets;
  const spaceIds = (_request$body$space_i = (_request$body3 = request.body) === null || _request$body3 === void 0 ? void 0 : _request$body3.space_ids) !== null && _request$body$space_i !== void 0 ? _request$body$space_i : [spaceId];
  for (const spaceToInstallId of spaceIds) {
    var _installedPkgWithAsse;
    await (0, _install.installKibanaAssetsAndReferences)({
      savedObjectsClient: _services.appContextService.getInternalUserSOClientForSpaceId(spaceToInstallId),
      logger,
      pkgName,
      pkgTitle: packageInfo.title,
      installAsAdditionalSpace: true,
      spaceId: spaceToInstallId,
      assetTags: (_installedPkgWithAsse = installedPkgWithAssets.packageInfo) === null || _installedPkgWithAsse === void 0 ? void 0 : _installedPkgWithAsse.asset_tags,
      installedPkg: installation,
      packageInstallContext: {
        packageInfo,
        paths: installedPkgWithAssets.paths,
        archiveIterator: (0, _archive_iterator.createArchiveIteratorFromMap)(installedPkgWithAssets.assetsMap)
      }
    });
  }
  return response.ok({
    body: {
      success: true
    }
  });
};
exports.installPackageKibanaAssetsHandler = installPackageKibanaAssetsHandler;
const deletePackageKibanaAssetsHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const savedObjectsClient = fleetContext.internalSoClient;
  const logger = _services.appContextService.getLogger();
  const spaceId = fleetContext.spaceId;
  const {
    pkgName,
    pkgVersion
  } = request.params;
  const installation = await (0, _get.getInstallationObject)({
    pkgName,
    savedObjectsClient
  });
  if (!installation || installation.attributes.version !== pkgVersion) {
    throw new _errors.FleetNotFoundError('Version is not installed');
  }
  await (0, _install.deleteKibanaAssetsAndReferencesForSpace)({
    savedObjectsClient,
    logger,
    pkgName,
    spaceId,
    installedPkg: installation
  });
  return response.ok({
    body: {
      success: true
    }
  });
};
exports.deletePackageKibanaAssetsHandler = deletePackageKibanaAssetsHandler;
const installRuleAssetsHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const savedObjectsClient = fleetContext.internalSoClient;
  const logger = _services.appContextService.getLogger();
  const spaceId = fleetContext.spaceId;
  const {
    pkgName,
    pkgVersion
  } = request.params;
  const installedPkgWithAssets = await (0, _get.getInstalledPackageWithAssets)({
    savedObjectsClient,
    pkgName,
    logger
  });
  const installation = await (0, _get.getInstallationObject)({
    pkgName,
    savedObjectsClient
  });
  if (!installation || !installedPkgWithAssets || (installedPkgWithAssets === null || installedPkgWithAssets === void 0 ? void 0 : installedPkgWithAssets.installation.version) !== pkgVersion) {
    throw new _errors.FleetNotFoundError('Requested version is not installed');
  }
  const user = _services.appContextService.getSecurityCore().authc.getCurrentUser(request) || undefined;
  const authorizationHeader = _http_authorization_header.HTTPAuthorizationHeader.parseFromRequest(request, user === null || user === void 0 ? void 0 : user.username);
  if (!authorizationHeader) {
    throw new _errors.FleetUnauthorizedError('Authorization header is missing or invalid');
  }
  const {
    packageInfo
  } = installedPkgWithAssets;
  await (0, _step_create_alerting_rules.stepCreateAlertingRules)({
    logger,
    savedObjectsClient,
    packageInstallContext: {
      packageInfo,
      paths: installedPkgWithAssets.paths,
      archiveIterator: (0, _archive_iterator.createArchiveIteratorFromMap)(installedPkgWithAssets.assetsMap)
    },
    spaceId,
    authorizationHeader
  });
  return response.ok({
    body: {
      success: true
    }
  });
};
exports.installRuleAssetsHandler = installRuleAssetsHandler;