"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.regexStringRt = exports.paginationSchema = exports.mimeTypeString = exports.limitedStringSchema = exports.limitedNumberSchema = exports.limitedNumberAsIntegerSchema = exports.limitedArraySchema = exports.NonEmptyString = void 0;
var rt = _interopRequireWildcard(require("io-ts"));
var _Either = require("fp-ts/Either");
var _constants = require("../constants");
var _types = require("./types");
var _mime_types = require("../constants/mime_types");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const NonEmptyString = exports.NonEmptyString = new rt.Type('NonEmptyString', rt.string.is, (input, context) => _Either.either.chain(rt.string.validate(input, context), s => {
  if (s.trim() !== '') {
    return rt.success(s);
  } else {
    return rt.failure(input, context, 'string must have length >= 1');
  }
}), rt.identity);
const limitedStringSchema = ({
  fieldName,
  min,
  max
}) => new rt.Type('LimitedString', rt.string.is, (input, context) => _Either.either.chain(rt.string.validate(input, context), s => {
  const trimmedString = s.trim();
  if (trimmedString.length === 0 && trimmedString.length < min) {
    return rt.failure(input, context, `The ${fieldName} field cannot be an empty string.`);
  }
  if (trimmedString.length < min) {
    return rt.failure(input, context, `The length of the ${fieldName} is too short. The minimum length is ${min}.`);
  }
  if (trimmedString.length > max) {
    return rt.failure(input, context, `The length of the ${fieldName} is too long. The maximum length is ${max}.`);
  }
  return rt.success(s);
}), rt.identity);
exports.limitedStringSchema = limitedStringSchema;
const limitedArraySchema = ({
  codec,
  fieldName,
  min,
  max
}) => new rt.Type('LimitedArray', input => rt.array(codec).is(input), (input, context) => _Either.either.chain(rt.array(codec).validate(input, context), s => {
  if (s.length < min) {
    return rt.failure(input, context, `The length of the field ${fieldName} is too short. Array must be of length >= ${min}.`);
  }
  if (s.length > max) {
    return rt.failure(input, context, `The length of the field ${fieldName} is too long. Array must be of length <= ${max}.`);
  }
  return rt.success(s);
}), rt.identity);
exports.limitedArraySchema = limitedArraySchema;
const paginationSchema = ({
  maxPerPage
}) => new rt.PartialType('Pagination', _types.PaginationSchemaRt.is, (u, c) => _Either.either.chain(_types.PaginationSchemaRt.validate(u, c), params => {
  var _params$page, _params$perPage;
  if (params.page == null && params.perPage == null) {
    return rt.success(params);
  }
  const pageAsNumber = (_params$page = params.page) !== null && _params$page !== void 0 ? _params$page : 0;
  const perPageAsNumber = (_params$perPage = params.perPage) !== null && _params$perPage !== void 0 ? _params$perPage : 0;
  if (perPageAsNumber > maxPerPage) {
    return rt.failure(u, c, `The provided perPage value is too high. The maximum allowed perPage value is ${maxPerPage}.`);
  }
  if (Math.max(pageAsNumber, pageAsNumber * perPageAsNumber) > _constants.MAX_DOCS_PER_PAGE) {
    return rt.failure(u, c, `The number of documents is too high. Paginating through more than ${_constants.MAX_DOCS_PER_PAGE} documents is not possible.`);
  }
  return rt.success({
    ...(params.page != null && {
      page: pageAsNumber
    }),
    ...(params.perPage != null && {
      perPage: perPageAsNumber
    })
  });
}), rt.identity, undefined);
exports.paginationSchema = paginationSchema;
const limitedNumberSchema = ({
  fieldName,
  min,
  max
}) => new rt.Type('LimitedNumber', rt.number.is, (input, context) => _Either.either.chain(rt.number.validate(input, context), s => {
  if (s < min) {
    return rt.failure(input, context, `The ${fieldName} field cannot be less than ${min}.`);
  }
  if (s > max) {
    return rt.failure(input, context, `The ${fieldName} field cannot be more than ${max}.`);
  }
  return rt.success(s);
}), rt.identity);
exports.limitedNumberSchema = limitedNumberSchema;
const limitedNumberAsIntegerSchema = ({
  fieldName
}) => new rt.Type('LimitedNumberAsInteger', rt.number.is, (input, context) => _Either.either.chain(rt.number.validate(input, context), s => {
  if (!Number.isSafeInteger(s)) {
    return rt.failure(input, context, `The ${fieldName} field should be an integer between -(2^53 - 1) and 2^53 - 1, inclusive.`);
  }
  return rt.success(s);
}), rt.identity);
exports.limitedNumberAsIntegerSchema = limitedNumberAsIntegerSchema;
const regexStringRt = ({
  codec,
  pattern,
  message
}) => new rt.Type('RegexString', codec.is, (input, context) => _Either.either.chain(codec.validate(input, context), value => {
  const regex = new RegExp(pattern, 'g');
  if (!regex.test(value)) {
    return rt.failure(input, context, message);
  }
  return rt.success(value);
}), rt.identity);
exports.regexStringRt = regexStringRt;
const mimeTypeString = exports.mimeTypeString = new rt.Type('mimeTypeString', rt.string.is, (input, context) => _Either.either.chain(rt.string.validate(input, context), s => {
  if (!_mime_types.ALLOWED_MIME_TYPES.includes(s)) {
    return rt.failure(input, context, `The mime type field value ${s} is not allowed.`);
  }
  return rt.success(s);
}), rt.identity);