"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useRoutingContext = void 0;
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _reactRedux = require("react-redux");
var _workpad = require("../../../state/selectors/workpad");
var _time_interval = require("../../../lib/time_interval");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useRoutingContext = () => {
  const [isAutoplayPaused, setIsAutoplayPaused] = (0, _react.useState)(false);
  const history = (0, _reactRouterDom.useHistory)();
  const {
    search
  } = history.location;
  const params = (0, _reactRouterDom.useParams)();
  const workpad = (0, _reactRedux.useSelector)(_workpad.getWorkpad);
  const searchParams = new URLSearchParams(search);
  const parsedPage = parseInt(params.pageNumber, 10);
  const pageNumber = isNaN(parsedPage) ? workpad.page + 1 : parsedPage;
  const workpadPages = workpad.pages.length;
  const getUrl = (0, _react.useCallback)(page => `/workpad/${params.id}/page/${page}${history.location.search}`, [params.id, history.location.search]);
  const gotoPage = (0, _react.useCallback)(page => {
    history.push(getUrl(page));
  }, [getUrl, history]);
  const nextPage = (0, _react.useCallback)(() => {
    let newPage = pageNumber + 1;
    if (newPage > workpadPages) {
      newPage = 1;
    }
    gotoPage(newPage);
  }, [pageNumber, workpadPages, gotoPage]);
  const previousPage = (0, _react.useCallback)(() => {
    let newPage = pageNumber - 1;
    if (newPage < 1) {
      newPage = workpadPages;
    }
    gotoPage(newPage);
  }, [pageNumber, workpadPages, gotoPage]);
  const isFullscreen = searchParams.get('__fullScreen') === 'true';
  const autoplayValue = searchParams.get('__autoplayInterval');
  const autoplayInterval = autoplayValue && (0, _time_interval.isValidTimeInterval)(autoplayValue) ? (0, _time_interval.getTimeInterval)(autoplayValue) || 0 : 0;
  const refreshValue = searchParams.get('__refreshInterval');
  const refreshInterval = refreshValue && (0, _time_interval.isValidTimeInterval)(refreshValue) ? (0, _time_interval.getTimeInterval)(refreshValue) || 0 : 0;
  const setFullscreen = (0, _react.useCallback)(enable => {
    const newQuery = new URLSearchParams(history.location.search);
    if (enable) {
      newQuery.set('__fullScreen', 'true');
    } else {
      setIsAutoplayPaused(false);
      newQuery.delete('__fullScreen');
    }
    history.push(`${history.location.pathname}?${newQuery.toString()}`);
  }, [history, setIsAutoplayPaused]);
  const setAutoplayInterval = (0, _react.useCallback)(interval => {
    const newQuery = new URLSearchParams(history.location.search);
    if (interval > 0) {
      newQuery.set('__autoplayInterval', (0, _time_interval.createTimeInterval)(interval));
    } else {
      newQuery.delete('__autoplayInterval');
    }
    history.push(`${history.location.pathname}?${newQuery.toString()}`);
  }, [history]);
  const setRefreshInterval = (0, _react.useCallback)(interval => {
    const newQuery = new URLSearchParams(history.location.search);
    if (interval > 0) {
      newQuery.set('__refreshInterval', (0, _time_interval.createTimeInterval)(interval));
    } else {
      newQuery.delete('__refreshInterval');
    }
    history.push(`${history.location.pathname}?${newQuery.toString()}`);
  }, [history]);
  const undo = (0, _react.useCallback)(() => {
    history.goBack();
  }, [history]);
  const redo = (0, _react.useCallback)(() => {
    history.goForward();
  }, [history]);
  const getRoutingContext = (0, _react.useCallback)(() => ({
    gotoPage,
    getUrl,
    isFullscreen,
    setFullscreen,
    autoplayInterval,
    setAutoplayInterval,
    nextPage,
    previousPage,
    refreshInterval,
    setRefreshInterval,
    isAutoplayPaused,
    setIsAutoplayPaused,
    undo,
    redo
  }), [gotoPage, getUrl, isFullscreen, setFullscreen, autoplayInterval, setAutoplayInterval, nextPage, previousPage, refreshInterval, setRefreshInterval, isAutoplayPaused, setIsAutoplayPaused, undo, redo]);
  return (0, _react.useMemo)(() => getRoutingContext(), [getRoutingContext]);
};
exports.useRoutingContext = useRoutingContext;