"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getComparisonOptionsForDiffOutcome = void 0;
exports.getOptionDetails = getOptionDetails;
exports.getVersionsForComparison = getVersionsForComparison;
exports.pickFieldValueForVersion = pickFieldValueForVersion;
exports.stringifyToSortedJson = void 0;
var _jsonStableStringify = _interopRequireDefault(require("json-stable-stringify"));
var _constants = require("./versions_picker/constants");
var _detection_engine = require("../../../../../../../common/api/detection_engine");
var _versions_picker = require("./versions_picker/versions_picker");
var _utility_types = require("../../../../../../../common/utility_types");
var i18n = _interopRequireWildcard(require("./translations"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Picks the field value for a given version either from a three-way diff object or from a user-set resolved value.
 *
 * @param version - The version for which the field value is to be picked.
 * @param fieldThreeWayDiff - The three-way diff object containing the field values for different versions.
 * @param resolvedValue - A value field will be upgraded to.
 * @returns - The field value for the specified version
 */
function pickFieldValueForVersion(version, fieldThreeWayDiff, resolvedValue) {
  if (version === _constants.Version.Final) {
    return resolvedValue;
  }
  const versionFieldToPick = `${version}_version`;
  return fieldThreeWayDiff[versionFieldToPick];
}

/**
 * Stringifies a field value to an alphabetically sorted JSON string.
 */
const stringifyToSortedJson = fieldValue => {
  if (fieldValue === undefined) {
    return '';
  }
  if (typeof fieldValue === 'string') {
    return fieldValue;
  }
  return (0, _jsonStableStringify.default)(fieldValue, {
    space: 2
  });
};
exports.stringifyToSortedJson = stringifyToSortedJson;
/**
 * Returns the title and description for a given versions picker option.
 */
function getOptionDetails(option) {
  switch (option) {
    case _versions_picker.VersionsPickerOptionEnum.MyChanges:
      return {
        title: i18n.MY_CHANGES_AND_FINAL_UPDATES_TITLE,
        description: i18n.MY_CHANGES_AND_FINAL_UPDATES_EXPLANATION
      };
    case _versions_picker.VersionsPickerOptionEnum.MyOriginalChanges:
      return {
        title: i18n.MY_ORIGINAL_CHANGES_TITLE,
        description: i18n.MY_ORIGINAL_CHANGES_EXPLANATION
      };
    case _versions_picker.VersionsPickerOptionEnum.UpdateFromElastic:
      return {
        title: i18n.UPDATE_FROM_ELASTIC_TITLE,
        description: i18n.UPDATE_FROM_ELASTIC_EXPLANATION
      };
    case _versions_picker.VersionsPickerOptionEnum.Merged:
      return {
        title: i18n.MERGED_CHANGES_TITLE,
        description: i18n.MERGED_CHANGES_EXPLANATION
      };
    default:
      return (0, _utility_types.assertUnreachable)(option);
  }
}

/**
 * Returns the versions to be compared based on the selected versions picker option.
 */
function getVersionsForComparison(selectedOption, hasBaseVersion) {
  switch (selectedOption) {
    case _versions_picker.VersionsPickerOptionEnum.MyChanges:
      return hasBaseVersion ? [_constants.Version.Base, _constants.Version.Final] : [_constants.Version.Current, _constants.Version.Final];
    case _versions_picker.VersionsPickerOptionEnum.MyOriginalChanges:
      return [_constants.Version.Base, _constants.Version.Current];
    case _versions_picker.VersionsPickerOptionEnum.UpdateFromElastic:
      return hasBaseVersion ? [_constants.Version.Base, _constants.Version.Target] : [_constants.Version.Current, _constants.Version.Target];
    case _versions_picker.VersionsPickerOptionEnum.Merged:
      return [_constants.Version.Base, _constants.Version.Final];
    default:
      return (0, _utility_types.assertUnreachable)(selectedOption);
  }
}

/**
 * Returns the versions picker options available for a given field diff outcome.
 */
const getComparisonOptionsForDiffOutcome = (diffOutcome, conflict, hasResolvedValueDifferentFromSuggested) => {
  switch (diffOutcome) {
    case _detection_engine.ThreeWayDiffOutcome.StockValueCanUpdate:
      {
        const options = [];
        if (hasResolvedValueDifferentFromSuggested) {
          options.push(_versions_picker.VersionsPickerOptionEnum.MyChanges);
        }
        options.push(_versions_picker.VersionsPickerOptionEnum.UpdateFromElastic);
        return options;
      }
    case _detection_engine.ThreeWayDiffOutcome.CustomizedValueNoUpdate:
      return [_versions_picker.VersionsPickerOptionEnum.MyChanges];
    case _detection_engine.ThreeWayDiffOutcome.CustomizedValueSameUpdate:
      return [_versions_picker.VersionsPickerOptionEnum.MyChanges, _versions_picker.VersionsPickerOptionEnum.UpdateFromElastic];
    case _detection_engine.ThreeWayDiffOutcome.CustomizedValueCanUpdate:
      {
        if (conflict === _detection_engine.ThreeWayDiffConflict.SOLVABLE) {
          return [hasResolvedValueDifferentFromSuggested ? _versions_picker.VersionsPickerOptionEnum.MyChanges : _versions_picker.VersionsPickerOptionEnum.Merged, _versions_picker.VersionsPickerOptionEnum.UpdateFromElastic, _versions_picker.VersionsPickerOptionEnum.MyOriginalChanges];
        }
        if (conflict === _detection_engine.ThreeWayDiffConflict.NON_SOLVABLE) {
          const options = [_versions_picker.VersionsPickerOptionEnum.MyChanges, _versions_picker.VersionsPickerOptionEnum.UpdateFromElastic];
          if (hasResolvedValueDifferentFromSuggested) {
            options.push(_versions_picker.VersionsPickerOptionEnum.MyOriginalChanges);
          }
          return options;
        }
      }
    case _detection_engine.ThreeWayDiffOutcome.MissingBaseCanUpdate:
      {
        const options = [];
        if (hasResolvedValueDifferentFromSuggested) {
          options.push(_versions_picker.VersionsPickerOptionEnum.MyChanges);
        }
        options.push(_versions_picker.VersionsPickerOptionEnum.UpdateFromElastic);
        return options;
      }
    default:
      return [];
  }
};
exports.getComparisonOptionsForDiffOutcome = getComparisonOptionsForDiffOutcome;