"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createListRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _zodHelpers = require("@kbn/zod-helpers");
var _api = require("@kbn/securitysolution-lists-common/api");
var _utils = require("../utils");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createListRoute = router => {
  router.versioned.post({
    access: 'public',
    path: _securitysolutionListConstants.LIST_URL,
    security: {
      authz: {
        requiredPrivileges: ['lists-all']
      }
    }
  }).addVersion({
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_api.CreateListRequestBody)
      }
    },
    version: '2023-10-31'
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const {
        name,
        description,
        deserializer,
        id,
        serializer,
        type,
        meta,
        version
      } = request.body;
      const lists = await (0, _.getListClient)(context);
      const dataStreamExists = await lists.getListDataStreamExists();
      const indexExists = await lists.getListIndexExists();
      if (!dataStreamExists && !indexExists) {
        return siemResponse.error({
          body: `To create a list, the data stream must exist first. Data stream "${lists.getListName()}" does not exist`,
          statusCode: 400
        });
      }

      // needs to be migrated to data stream
      if (!dataStreamExists && indexExists) {
        await lists.migrateListIndexToDataStream();
      }
      if (id != null) {
        const list = await lists.getList({
          id
        });
        if (list != null) {
          return siemResponse.error({
            body: `list id: "${id}" already exists`,
            statusCode: 409
          });
        }
      }
      const list = await lists.createList({
        description,
        deserializer,
        id,
        immutable: false,
        meta,
        name,
        serializer,
        type,
        version
      });
      return response.ok({
        body: _api.CreateListResponse.parse(list)
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.createListRoute = createListRoute;