"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getColumnTimeShiftWarnings = getColumnTimeShiftWarnings;
exports.getDateHistogramInterval = getDateHistogramInterval;
exports.getLayerTimeShiftChecks = getLayerTimeShiftChecks;
exports.getStateTimeShiftWarningMessages = getStateTimeShiftWarningMessages;
exports.parseTimeShiftWrapper = parseTimeShiftWrapper;
exports.resolveTimeShift = resolveTimeShift;
exports.timeShiftOptions = exports.timeShiftOptionOrder = void 0;
var _i18n = require("@kbn/i18n");
var _react = _interopRequireDefault(require("react"));
var _i18nReact = require("@kbn/i18n-react");
var _moment = _interopRequireDefault(require("moment"));
var _public = require("@kbn/data-plugin/public");
var _common = require("@kbn/data-plugin/common");
var _user_messages_ids = require("../../user_messages_ids");
var _jsxFileName = "/opt/buildkite-agent/builds/bk-agent-prod-gcp-1762971603108095020/elastic/kibana-artifacts-snapshot/kibana/x-pack/platform/plugins/shared/lens/public/datasources/form_based/time_shift_utils.tsx";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
function parseTimeShiftWrapper(timeShiftString, dateRange) {
  if ((0, _common.isAbsoluteTimeShift)(timeShiftString.trim())) {
    return (0, _common.parseAbsoluteTimeShift)(timeShiftString, {
      from: dateRange.fromDate,
      to: dateRange.toDate
    }).value;
  }
  return (0, _common.parseTimeShift)(timeShiftString);
}
const timeShiftOptions = exports.timeShiftOptions = [{
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.none', {
    defaultMessage: 'None'
  }),
  value: ''
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.hour', {
    defaultMessage: '1 hour ago (1h)'
  }),
  value: '1h'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.3hours', {
    defaultMessage: '3 hours ago (3h)'
  }),
  value: '3h'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.6hours', {
    defaultMessage: '6 hours ago (6h)'
  }),
  value: '6h'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.12hours', {
    defaultMessage: '12 hours ago (12h)'
  }),
  value: '12h'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.day', {
    defaultMessage: '1 day ago (1d)'
  }),
  value: '1d'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.week', {
    defaultMessage: '1 week ago (1w)'
  }),
  value: '1w'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.month', {
    defaultMessage: '1 month ago (1M)'
  }),
  value: '1M'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.3months', {
    defaultMessage: '3 months ago (3M)'
  }),
  value: '3M'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.6months', {
    defaultMessage: '6 months ago (6M)'
  }),
  value: '6M'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.year', {
    defaultMessage: '1 year ago (1y)'
  }),
  value: '1y'
}, {
  label: _i18n.i18n.translate('xpack.lens.indexPattern.timeShift.previous', {
    defaultMessage: 'Previous time range'
  }),
  value: 'previous'
}];
const timeShiftOptionOrder = exports.timeShiftOptionOrder = timeShiftOptions.reduce((optionMap, {
  value
}, index) => ({
  ...optionMap,
  [value]: index
}), {});
function getDateHistogramInterval(datatableUtilities, layer, indexPattern, activeData, layerId) {
  const dateHistogramColumn = layer.columnOrder.find(colId => layer.columns[colId].operationType === 'date_histogram');
  if (!dateHistogramColumn && !indexPattern.timeFieldName) {
    return {
      canShift: false,
      hasDateHistogram: false
    };
  }
  if (dateHistogramColumn && activeData && activeData[layerId] && activeData[layerId]) {
    const column = activeData[layerId].columns.find(col => col.id === dateHistogramColumn);
    if (column) {
      var _datatableUtilities$g;
      const expression = ((_datatableUtilities$g = datatableUtilities.getDateHistogramMeta(column)) === null || _datatableUtilities$g === void 0 ? void 0 : _datatableUtilities$g.interval) || '';
      return {
        interval: _public.search.aggs.parseInterval(expression),
        expression,
        canShift: true,
        hasDateHistogram: true
      };
    }
  }
  return {
    canShift: true,
    hasDateHistogram: Boolean(dateHistogramColumn)
  };
}
function getLayerTimeShiftChecks({
  interval: dateHistogramInterval,
  canShift
}) {
  return {
    canShift,
    isValueTooSmall: parsedValue => {
      return dateHistogramInterval && parsedValue && typeof parsedValue === 'object' && parsedValue.asMilliseconds() < dateHistogramInterval.asMilliseconds();
    },
    isValueNotMultiple: parsedValue => {
      return dateHistogramInterval && parsedValue && typeof parsedValue === 'object' && !Number.isInteger(parsedValue.asMilliseconds() / dateHistogramInterval.asMilliseconds());
    },
    isInvalid: parsedValue => {
      return Boolean(parsedValue === 'invalid');
    }
  };
}
function getStateTimeShiftWarningMessages(datatableUtilities, state, {
  activeData,
  dataViews
}) {
  if (!state) return [];
  const warningMessages = [];
  Object.entries(state.layers).forEach(([layerId, layer]) => {
    const layerIndexPattern = dataViews.indexPatterns[layer.indexPatternId];
    if (!layerIndexPattern) {
      return;
    }
    const dateHistogramInterval = getDateHistogramInterval(datatableUtilities, layer, layerIndexPattern, activeData, layerId);
    if (!dateHistogramInterval.interval) {
      return;
    }
    const dateHistogramIntervalExpression = dateHistogramInterval.expression;
    const shiftInterval = dateHistogramInterval.interval.asMilliseconds();
    const timeShifts = new Set();
    const timeShiftMap = {};
    Object.entries(layer.columns).forEach(([columnId, column]) => {
      // TODO: I believe this can be replaced with a similar code like getColumnTimeShiftWarnings
      if (column.isBucketed) return;
      let duration = 0;
      // skip absolute time shifts as underneath it will be converted to be round
      // and avoid this type of issues
      if (column.timeShift && !(0, _common.isAbsoluteTimeShift)(column.timeShift)) {
        const parsedTimeShift = (0, _common.parseTimeShift)(column.timeShift);
        if (parsedTimeShift === 'previous' || parsedTimeShift === 'invalid') {
          return;
        }
        duration = parsedTimeShift.asMilliseconds();
      }
      timeShifts.add(duration);
      if (!timeShiftMap[duration]) {
        timeShiftMap[duration] = [];
      }
      timeShiftMap[duration].push(columnId);
    });
    if (timeShifts.size < 2) {
      return;
    }
    timeShifts.forEach(timeShift => {
      if (timeShift === 0) return;
      if (timeShift < shiftInterval) {
        timeShiftMap[timeShift].forEach(columnId => {
          warningMessages.push({
            uniqueId: _user_messages_ids.TIMESHIFT_LT_INTERVAL,
            severity: 'warning',
            fixableInEditor: true,
            displayLocations: [{
              id: 'toolbar'
            }],
            shortMessage: '',
            longMessage: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
              key: `small-${columnId}`,
              id: "xpack.lens.indexPattern.timeShiftSmallWarning",
              defaultMessage: "{label} uses a time shift of {columnTimeShift} which is smaller than the date histogram interval of {interval}. To prevent mismatched data, use a multiple of {interval} as time shift.",
              values: {
                label: /*#__PURE__*/_react.default.createElement("strong", {
                  __self: this,
                  __source: {
                    fileName: _jsxFileName,
                    lineNumber: 241,
                    columnNumber: 26
                  }
                }, layer.columns[columnId].label),
                interval: /*#__PURE__*/_react.default.createElement("strong", {
                  __self: this,
                  __source: {
                    fileName: _jsxFileName,
                    lineNumber: 242,
                    columnNumber: 29
                  }
                }, dateHistogramIntervalExpression),
                columnTimeShift: /*#__PURE__*/_react.default.createElement("strong", {
                  __self: this,
                  __source: {
                    fileName: _jsxFileName,
                    lineNumber: 243,
                    columnNumber: 36
                  }
                }, layer.columns[columnId].timeShift)
              },
              __self: this,
              __source: {
                fileName: _jsxFileName,
                lineNumber: 236,
                columnNumber: 15
              }
            })
          });
        });
      } else if (!Number.isInteger(timeShift / shiftInterval)) {
        timeShiftMap[timeShift].forEach(columnId => {
          warningMessages.push({
            uniqueId: _user_messages_ids.TIMESHIFT_NOT_MULTIPLE_INTERVAL,
            severity: 'warning',
            fixableInEditor: true,
            displayLocations: [{
              id: 'toolbar'
            }],
            shortMessage: '',
            longMessage: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
              key: `multiple-${columnId}`,
              id: "xpack.lens.indexPattern.timeShiftMultipleWarning",
              defaultMessage: "{label} uses a time shift of {columnTimeShift} which is not a multiple of the date histogram interval of {interval}. To prevent mismatched data, use a multiple of {interval} as time shift.",
              values: {
                label: /*#__PURE__*/_react.default.createElement("strong", {
                  __self: this,
                  __source: {
                    fileName: _jsxFileName,
                    lineNumber: 263,
                    columnNumber: 26
                  }
                }, layer.columns[columnId].label),
                interval: /*#__PURE__*/_react.default.createElement("strong", {
                  __self: this,
                  __source: {
                    fileName: _jsxFileName,
                    lineNumber: 264,
                    columnNumber: 29
                  }
                }, dateHistogramIntervalExpression),
                columnTimeShift: /*#__PURE__*/_react.default.createElement("strong", {
                  __self: this,
                  __source: {
                    fileName: _jsxFileName,
                    lineNumber: 265,
                    columnNumber: 36
                  }
                }, layer.columns[columnId].timeShift)
              },
              __self: this,
              __source: {
                fileName: _jsxFileName,
                lineNumber: 258,
                columnNumber: 15
              }
            })
          });
        });
      }
    });
  });
  return warningMessages;
}
function getColumnTimeShiftWarnings(dateHistogramInterval, timeShift) {
  const {
    isValueTooSmall,
    isValueNotMultiple
  } = getLayerTimeShiftChecks(dateHistogramInterval);
  const warnings = [];
  if ((0, _common.isAbsoluteTimeShift)(timeShift)) {
    return [];
  }
  const parsedLocalValue = timeShift && (0, _common.parseTimeShift)(timeShift);
  const localValueTooSmall = parsedLocalValue && isValueTooSmall(parsedLocalValue);
  const localValueNotMultiple = parsedLocalValue && isValueNotMultiple(parsedLocalValue);
  if (localValueTooSmall) {
    warnings.push(_i18n.i18n.translate('xpack.lens.indexPattern.timeShift.tooSmallHelp', {
      defaultMessage: 'Time shift should to be larger than the date histogram interval. Either increase time shift or specify smaller interval in date histogram'
    }));
  } else if (localValueNotMultiple) {
    warnings.push(_i18n.i18n.translate('xpack.lens.indexPattern.timeShift.noMultipleHelp', {
      defaultMessage: 'Time shift should be a multiple of the date histogram interval. Either adjust time shift or date histogram interval'
    }));
  }
  return warnings;
}
function closestMultipleOfInterval(duration, interval) {
  if (duration % interval === 0) {
    return duration;
  }
  return Math.ceil(duration / interval) * interval;
}
function roundAbsoluteInterval(timeShift, dateRange, targetBars) {
  // workout the interval (most probably matching the ES one)
  const interval = (0, _common.calcAutoIntervalNear)(targetBars, (0, _moment.default)(dateRange.toDate).diff((0, _moment.default)(dateRange.fromDate)));
  const duration = parseTimeShiftWrapper(timeShift, dateRange);
  if (typeof duration !== 'string') {
    const roundingOffset = timeShift.startsWith('end') ? interval.asMilliseconds() : 0;
    return `${(closestMultipleOfInterval(duration.asMilliseconds(), interval.asMilliseconds()) - roundingOffset) / 1000}s`;
  }
}
function resolveTimeShift(timeShift, dateRange, targetBars, hasDateHistogram = false) {
  if (timeShift && (0, _common.isAbsoluteTimeShift)(timeShift)) {
    return roundAbsoluteInterval(timeShift, dateRange, targetBars);
  }
  // Translate a relative "previous" shift into an absolute endAt(<current range start timestamp>)
  if (timeShift && hasDateHistogram && timeShift === 'previous') {
    return roundAbsoluteInterval(`endAt(${dateRange.fromDate})`, dateRange, targetBars);
  }
  return timeShift;
}