"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.analyzeFile = analyzeFile;
var _fileUploadCommon = require("@kbn/file-upload-common");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PREVIEW_DOC_LIMIT = 20;
async function analyzeFile(client, logger, data, overrides, includePreview) {
  overrides.explain = overrides.explain === undefined ? 'true' : overrides.explain;
  const results = await client.asInternalUser.textStructure.findStructure({
    body: data,
    ecs_compatibility: 'v1',
    ...overrides
  }, {
    maxRetries: 0
  });
  const {
    hasOverrides,
    reducedOverrides
  } = formatOverrides(overrides);
  let previewDocs;
  if (includePreview) {
    try {
      const pipeline = (0, _lodash.cloneDeep)(results.ingest_pipeline);
      (0, _fileUploadCommon.updatePipelineTimezone)(pipeline);
      const reader = getReader(results);
      const arrayBuffer = new Uint8Array(Buffer.from(data));
      const docs = reader.read(arrayBuffer).slice(0, PREVIEW_DOC_LIMIT);
      if (results.format === _fileUploadCommon.FILE_FORMATS.NDJSON) {
        previewDocs = {
          docs: docs.map(doc => ({
            doc: {
              _id: '',
              _index: '',
              _source: JSON.parse(doc)
            }
          }))
        };
      } else {
        previewDocs = await client.asInternalUser.ingest.simulate({
          pipeline,
          docs: docs.map(doc => {
            return {
              _source: doc
            };
          })
        });
      }
    } catch (error) {
      // preview failed, just log the error
      logger.warn(`Unable to generate preview documents, error: ${error.message}`);
    }
  }
  return {
    ...(hasOverrides && {
      overrides: reducedOverrides
    }),
    // @ts-expect-error type incompatible with FindFileStructureResponse
    results,
    preview: previewDocs
  };
}
function formatOverrides(overrides) {
  let hasOverrides = false;
  const reducedOverrides = Object.keys(overrides).reduce((acc, overrideKey) => {
    const overrideValue = overrides[overrideKey];
    if (overrideValue !== undefined && overrideValue !== '') {
      if (overrideKey === 'column_names') {
        acc.column_names = overrideValue.split(',');
      } else if (overrideKey === 'has_header_row') {
        acc.has_header_row = overrideValue === 'true';
      } else if (overrideKey === 'should_trim_fields') {
        acc.should_trim_fields = overrideValue === 'true';
      } else {
        acc[overrideKey] = overrideValue;
      }
      hasOverrides = true;
    }
    return acc;
  }, {});
  return {
    reducedOverrides,
    hasOverrides
  };
}
function getReader(results) {
  switch (results.format) {
    case _fileUploadCommon.FILE_FORMATS.NDJSON:
      return new _fileUploadCommon.NdjsonReader();
    case _fileUploadCommon.FILE_FORMATS.SEMI_STRUCTURED_TEXT:
    case _fileUploadCommon.FILE_FORMATS.DELIMITED:
      const options = {};
      if (results.exclude_lines_pattern !== undefined) {
        options.excludeLinesPattern = results.exclude_lines_pattern;
      }
      if (results.multiline_start_pattern !== undefined) {
        options.multilineStartPattern = results.multiline_start_pattern;
      }
      return new _fileUploadCommon.MessageReader(options);
    case _fileUploadCommon.FILE_FORMATS.TIKA:
      return new _fileUploadCommon.TikaReader();
    default:
      throw new Error(`Unknown format: ${results.format}`);
  }
}