"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CoreAppsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _querystring = require("querystring");
var _configSchema = require("@kbn/config-schema");
var _repoInfo = require("@kbn/repo-info");
var _rxjs = require("rxjs");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _core_app_config = require("./core_app_config");
var _bundle_routes = require("./bundle_routes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** @internal */

const DYNAMIC_CONFIG_OVERRIDES_SO_TYPE = 'dynamic-config-overrides';
const DYNAMIC_CONFIG_OVERRIDES_SO_ID = 'dynamic-config-overrides';

/** @internal */
class CoreAppsService {
  constructor(core) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "env", void 0);
    (0, _defineProperty2.default)(this, "configService", void 0);
    (0, _defineProperty2.default)(this, "config$", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsStart$", new _rxjs.ReplaySubject(1));
    (0, _defineProperty2.default)(this, "stop$", new _rxjs.Subject());
    this.logger = core.logger.get('core-app');
    this.env = core.env;
    this.configService = core.configService;
    this.config$ = this.configService.atPath(_core_app_config.CoreAppPath).pipe((0, _rxjs.map)(rawCfg => new _core_app_config.CoreAppConfig(rawCfg)));
  }
  async preboot(corePreboot, uiPlugins) {
    this.logger.debug('Prebooting core app.');

    // We register app-serving routes only if there are `preboot` plugins that may need them.
    if (uiPlugins.public.size > 0) {
      const config = await (0, _rxjs.firstValueFrom)(this.config$);
      this.registerPrebootDefaultRoutes(corePreboot, uiPlugins, config.skipBundleRoutesIfCdnEnabled && corePreboot.http.staticAssets.isUsingCdn());
      this.registerStaticDirs(corePreboot, uiPlugins);
    }
  }
  async setup(coreSetup, uiPlugins) {
    this.logger.debug('Setting up core app.');
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    this.registerDefaultRoutes(coreSetup, uiPlugins, config.skipBundleRoutesIfCdnEnabled && coreSetup.http.staticAssets.isUsingCdn());
    this.registerStaticDirs(coreSetup, uiPlugins);
    this.maybeRegisterDynamicConfigurationFeature({
      config,
      coreSetup,
      savedObjectsStart$: this.savedObjectsStart$
    });
  }
  start(coreStart) {
    this.savedObjectsStart$.next(coreStart.savedObjects);
  }
  stop() {
    this.stop$.next();
  }
  registerPrebootDefaultRoutes(corePreboot, uiPlugins, skipBundleRegistration) {
    corePreboot.http.registerRoutes('', router => {
      this.registerCommonDefaultRoutes({
        basePath: corePreboot.http.basePath,
        httpResources: corePreboot.httpResources.createRegistrar(router),
        staticAssets: corePreboot.http.staticAssets,
        router,
        uiPlugins,
        onResourceNotFound: async (req, res) =>
        // The API consumers might call various Kibana APIs (e.g. `/api/status`) when Kibana is still at the preboot
        // stage, and the main HTTP server that registers API handlers isn't up yet. At this stage we don't know if
        // the API endpoint exists or not, and hence cannot reply with `404`. We also should not reply with completely
        // unexpected response (`200 text/html` for the Core app). The only suitable option is to reply with `503`
        // like we do for all other unknown non-GET requests at the preboot stage.
        req.route.path.startsWith('/api/') || req.route.path.startsWith('/internal/') ? res.customError({
          statusCode: 503,
          headers: {
            'Retry-After': '30'
          },
          body: 'Kibana server is not ready yet',
          bypassErrorFormat: true
        }) : res.renderCoreApp(),
        skipBundleRegistration
      });
    });
  }
  registerDefaultRoutes(coreSetup, uiPlugins, skipBundleRegistration) {
    const httpSetup = coreSetup.http;
    const router = httpSetup.createRouter('');
    const resources = coreSetup.httpResources.createRegistrar(router);
    router.get({
      path: '/',
      validate: false,
      options: {
        access: 'public'
      },
      security: {
        authz: {
          enabled: false,
          reason: 'This route is only used for serving the default route.'
        }
      }
    }, async (context, req, res) => {
      const {
        uiSettings
      } = await context.core;
      let defaultRoute = await uiSettings.client.get('defaultRoute', {
        request: req
      });
      if (!defaultRoute) {
        defaultRoute = '/app/home';
      }
      const basePath = httpSetup.basePath.get(req);
      const url = `${basePath}${defaultRoute}`;
      return res.redirected({
        headers: {
          location: url
        }
      });
    });
    this.registerCommonDefaultRoutes({
      basePath: coreSetup.http.basePath,
      httpResources: resources,
      staticAssets: coreSetup.http.staticAssets,
      router,
      uiPlugins,
      onResourceNotFound: async (req, res) => res.notFound(),
      skipBundleRegistration
    });
    resources.register({
      path: '/app/{id}/{any*}',
      validate: false,
      options: {
        excludeFromRateLimiter: true
      },
      security: {
        authz: {
          enabled: false,
          reason: 'The route is opted out of the authorization since it is a wrapper around core app view'
        },
        authc: {
          enabled: true
        }
      }
    }, async (context, request, response) => {
      return response.renderCoreApp();
    });
    const anonymousStatusPage = coreSetup.status.isStatusPageAnonymous();
    resources.register({
      path: '/status',
      validate: false,
      security: {
        authz: {
          enabled: false,
          reason: 'The route is opted out of the authorization since it is a wrapper around core app view'
        },
        authc: anonymousStatusPage ? {
          enabled: false,
          reason: 'The route is opted out of the authentication since it since it is a wrapper around core app anonymous view'
        } : {
          enabled: true
        }
      }
    }, async (context, request, response) => {
      if (anonymousStatusPage) {
        return response.renderAnonymousCoreApp();
      } else {
        return response.renderCoreApp();
      }
    });
  }
  maybeRegisterDynamicConfigurationFeature({
    config,
    coreSetup,
    savedObjectsStart$
  }) {
    // Always registering the Saved Objects to avoid ON/OFF conflicts in the migrations
    coreSetup.savedObjects.registerType({
      name: DYNAMIC_CONFIG_OVERRIDES_SO_TYPE,
      hidden: true,
      hiddenFromHttpApis: true,
      namespaceType: 'agnostic',
      mappings: {
        dynamic: false,
        properties: {}
      }
    });
    if (config.allowDynamicConfigOverrides) {
      const savedObjectsClient$ = savedObjectsStart$.pipe((0, _rxjs.map)(savedObjectsStart => savedObjectsStart.createInternalRepository([DYNAMIC_CONFIG_OVERRIDES_SO_TYPE])), (0, _rxjs.shareReplay)(1));

      // Register the HTTP route
      const router = coreSetup.http.createRouter('');
      this.registerInternalCoreSettingsRoute(router, savedObjectsClient$);
      let latestOverrideVersion; // Use the document version to avoid calling override on every poll
      // Poll for updates
      (0, _rxjs.combineLatest)([savedObjectsClient$, (0, _rxjs.timer)(0, 10_000)]).pipe((0, _rxjs.switchMap)(async ([soClient]) => {
        try {
          const persistedOverrides = await soClient.get(DYNAMIC_CONFIG_OVERRIDES_SO_TYPE, DYNAMIC_CONFIG_OVERRIDES_SO_ID);
          if (latestOverrideVersion !== persistedOverrides.version) {
            this.configService.setDynamicConfigOverrides(persistedOverrides.attributes);
            latestOverrideVersion = persistedOverrides.version;
            this.logger.info('Succeeded in applying persisted dynamic config overrides');
          }
        } catch (err) {
          // Potential failures:
          // - The SO document does not exist (404 error) => no need to log
          // - The configuration overrides are invalid => they won't be applied and the validation error will be logged.
          if (!_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(err)) {
            this.logger.warn(`Failed to apply the persisted dynamic config overrides: ${err}`);
          }
        }
      }), (0, _rxjs.takeUntil)(this.stop$)).subscribe();
    }
  }

  /**
   * Registers the HTTP API that allows updating in-memory the settings that opted-in to be dynamically updatable.
   * @param router {@link IRouter}
   * @param savedObjectClient$ An observable of a {@link SavedObjectsClientContract | savedObjects client} that will be used to update the document
   * @internal
   */
  registerInternalCoreSettingsRoute(router, savedObjectClient$) {
    router.versioned.put({
      path: '/internal/core/_settings',
      access: 'internal',
      security: {
        authz: {
          requiredPrivileges: ['updateDynamicConfig']
        }
      }
    }).addVersion({
      version: '1',
      validate: {
        request: {
          body: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any())
        },
        response: {
          '200': {
            body: () => _configSchema.schema.object({
              ok: _configSchema.schema.boolean()
            })
          }
        }
      }
    }, async (context, req, res) => {
      try {
        const newGlobalOverrides = this.configService.setDynamicConfigOverrides(req.body);
        const soClient = await (0, _rxjs.firstValueFrom)(savedObjectClient$);
        await soClient.create(DYNAMIC_CONFIG_OVERRIDES_SO_TYPE, newGlobalOverrides, {
          id: DYNAMIC_CONFIG_OVERRIDES_SO_ID,
          overwrite: true,
          refresh: false
        });
        // set it again in memory in case the timer polling the SO for updates has overridden it during this update.
        this.configService.setDynamicConfigOverrides(req.body);
      } catch (err) {
        if (err instanceof _configSchema.ValidationError) {
          return res.badRequest({
            body: err
          });
        }
        throw err;
      }
      return res.ok({
        body: {
          ok: true
        }
      });
    });
  }
  registerCommonDefaultRoutes({
    router,
    basePath,
    staticAssets,
    uiPlugins,
    onResourceNotFound,
    httpResources,
    skipBundleRegistration
  }) {
    // catch-all route
    httpResources.register({
      path: '/{path*}',
      validate: {
        params: _configSchema.schema.object({
          path: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        query: _configSchema.schema.maybe(_configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any()))
      },
      security: {
        authz: {
          enabled: false,
          reason: 'The route is opted out of the authorization since it is a catch-all route'
        }
      }
    }, async (context, req, res) => {
      const {
        query,
        params
      } = req;
      const {
        path
      } = params;
      if (!path || !path.endsWith('/') || path.startsWith('/')) {
        return onResourceNotFound(req, res);
      }

      // remove trailing slash
      const requestBasePath = basePath.get(req);
      let rewrittenPath = path.slice(0, -1);
      if (`/${path}`.startsWith(requestBasePath)) {
        rewrittenPath = rewrittenPath.substring(requestBasePath.length);
      }
      const querystring = query ? (0, _querystring.stringify)(query) : undefined;
      const url = `${requestBasePath}/${encodeURIComponent(rewrittenPath)}${querystring ? `?${querystring}` : ''}`;
      return res.redirected({
        headers: {
          location: url
        }
      });
    });
    router.get({
      path: '/core',
      validate: false,
      security: {
        authz: {
          enabled: false,
          reason: 'The route is opted out of the authorization since it returns static response'
        }
      }
    }, async (context, req, res) => res.ok({
      body: {
        version: '0.0.1'
      }
    }));
    if (!skipBundleRegistration) {
      (0, _bundle_routes.registerBundleRoutes)({
        router,
        uiPlugins,
        staticAssets,
        packageInfo: this.env.packageInfo
      });
    }
  }

  // After the package is built and bootstrap extracts files to target/build,
  // assets are exposed at the root of the package and in the package's node_modules dir
  registerStaticDirs(core, uiPlugins) {
    // Expose @elastic/charts' pre-compiled CSS themes
    // Referenced in `getThemeStylesheetPaths` in src/core/packages/rendering/server-internal/src/render_utils.ts
    core.http.registerStaticDir(
    // We're only interested in exposing 'theme_dark.css' and 'theme_light.css',
    // but this route cannot be limited to a file (not even a pattern like `{file}.css`).
    // We don't think that it's too risky because @elastic/charts is a public code.
    // Still, it's limited to the root dir (missing * in the pattern) for extra security.
    core.http.staticAssets.prependServerPath(`/ui/charts/{file}`), (0, _repoInfo.fromRoot)(`node_modules/@elastic/charts/dist/`));
    // });

    /**
     * Serve UI from sha-scoped and not-sha-scoped paths to allow time for plugin code to migrate
     * Eventually we only want to serve from the sha scoped path
     */
    [core.http.staticAssets.prependServerPath('/ui/{path*}'), '/ui/{path*}'].forEach(path => {
      core.http.registerStaticDir(path, (0, _repoInfo.fromRoot)('node_modules/@kbn/core-apps-server-internal/assets'));
    });
    for (const [pluginName, pluginInfo] of uiPlugins.internal) {
      if (!pluginInfo.publicAssetsDir) continue;
      /**
       * Serve UI from sha-scoped and not-sha-scoped paths to allow time for plugin code to migrate
       * Eventually we only want to serve from the sha scoped path
       */
      [core.http.staticAssets.getPluginServerPath(pluginName, '{path*}'), `/plugins/${pluginName}/assets/{path*}`].forEach(path => {
        core.http.registerStaticDir(path, pluginInfo.publicAssetsDir);
      });
    }
  }
}
exports.CoreAppsService = CoreAppsService;