"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MaintenanceWindowsService = exports.DEFAULT_CACHE_INTERVAL_MS = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _get_maintenance_windows = require("./get_maintenance_windows");
var _lib = require("../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_CACHE_INTERVAL_MS = exports.DEFAULT_CACHE_INTERVAL_MS = 60000; // 1 minute cache

class MaintenanceWindowsService {
  constructor(options) {
    (0, _defineProperty2.default)(this, "cacheIntervalMs", DEFAULT_CACHE_INTERVAL_MS);
    (0, _defineProperty2.default)(this, "windows", new Map());
    this.options = options;
    if (options.cacheInterval) {
      this.cacheIntervalMs = options.cacheInterval;
    }
  }
  async getMaintenanceWindows(opts) {
    const activeMaintenanceWindows = await this.loadMaintenanceWindows({
      request: opts.request,
      spaceId: opts.spaceId
    });

    // Filter maintenance windows on current time
    const now = Date.now();
    const currentlyActiveMaintenanceWindows = activeMaintenanceWindows.filter(mw => mw.events.some(event => new Date(event.gte).getTime() <= now && now <= new Date(event.lte).getTime()));

    // Only look at maintenance windows for this rule category
    const maintenanceWindows = currentlyActiveMaintenanceWindows.filter(({
      categoryIds
    }) => {
      // If category IDs array doesn't exist: allow all
      if (!Array.isArray(categoryIds)) {
        return true;
      }
      // If category IDs array exist: check category
      if (categoryIds.includes(opts.ruleTypeCategory)) {
        return true;
      }
      return false;
    });

    // Set the event log MW Id field the first time with MWs without scoped queries
    const maintenanceWindowsWithoutScopedQueryIds = (0, _get_maintenance_windows.filterMaintenanceWindowsIds)({
      maintenanceWindows,
      withScopedQuery: false
    });
    if (maintenanceWindowsWithoutScopedQueryIds.length) {
      opts.eventLogger.setMaintenanceWindowIds(maintenanceWindowsWithoutScopedQueryIds);
    }
    return {
      maintenanceWindows,
      maintenanceWindowsWithoutScopedQueryIds
    };
  }
  async loadMaintenanceWindows(opts) {
    const now = Date.now();
    if (this.windows.has(opts.spaceId)) {
      const windowsFromLastUpdate = this.windows.get(opts.spaceId);
      const lastUpdated = new Date(windowsFromLastUpdate.lastUpdated).getTime();
      if (now - lastUpdated >= this.cacheIntervalMs) {
        // cache expired, refetch settings
        try {
          return await this.fetchMaintenanceWindows(opts.request, opts.spaceId, now);
        } catch (err) {
          // return cached settings on error
          this.options.logger.debug(`Failed to fetch maintenance windows after cache expiration, using cached windows: ${err.message}`);
          return windowsFromLastUpdate.activeMaintenanceWindows;
        }
      } else {
        return windowsFromLastUpdate.activeMaintenanceWindows;
      }
    } else {
      // nothing in cache, fetch settings
      try {
        return await this.fetchMaintenanceWindows(opts.request, opts.spaceId, now);
      } catch (err) {
        // return default settings on error
        this.options.logger.debug(`Failed to fetch initial maintenance windows: ${err.message}`);
        return [];
      }
    }
  }
  async fetchMaintenanceWindows(request, spaceId, now) {
    return await (0, _lib.withAlertingSpan)('alerting:load-maintenance-windows', async () => {
      const maintenanceWindowClient = this.options.getMaintenanceWindowClientWithRequest(request);
      const activeMaintenanceWindows = await maintenanceWindowClient.getActiveMaintenanceWindows(this.cacheIntervalMs);
      this.windows.set(spaceId, {
        lastUpdated: now,
        activeMaintenanceWindows
      });
      return activeMaintenanceWindows;
    });
  }
}
exports.MaintenanceWindowsService = MaintenanceWindowsService;