"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatError = formatError;
exports.formatStack = formatStack;
var _util = require("util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getErrorMessage(error) {
  if (typeof error === 'string') {
    return error;
  }
  if (!error) {
    // stringify undefined/null/0/whatever this falsy value is
    return (0, _util.inspect)(error);
  }

  // handle es error response with `root_cause`s
  if (error.resp && error.resp.error && error.resp.error.root_cause) {
    return error.resp.error.root_cause.map(cause => cause.reason).join('\n');
  }

  // handle http response errors with error messages
  if (error.body && typeof error.body.message === 'string') {
    return error.body.message;
  }

  // handle standard error objects with messages
  if (error instanceof Error && error.message) {
    return error.message;
  }

  // everything else can just be serialized using util.inspect()
  return (0, _util.inspect)(error);
}

/**
 * Produce a string version of an error,
 */
function formatError(error, source) {
  return `${source ? source + ': ' : ''}${getErrorMessage(error)}`;
}

/**
 * Format the stack trace from a message so that it setups with the message, which
 * some browsers do automatically and some don't
 */
function formatStack(error) {
  var _error$stack;
  if (typeof error === 'string') {
    return '';
  }
  if (error.stack && !error.stack.includes(error.message)) {
    return 'Error: ' + error.message + '\n' + error.stack;
  }
  return (_error$stack = error.stack) !== null && _error$stack !== void 0 ? _error$stack : '';
}