"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStreamAdapter = getDataStreamAdapter;
var _lodash = require("lodash");
var _create_concrete_write_index = require("./create_concrete_write_index");
var _retry_transient_es_errors = require("./retry_transient_es_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// eslint-disable-next-line max-classes-per-file

function getDataStreamAdapter(opts) {
  if (opts.useDataStreamForAlerts) {
    return new DataStreamImplementation();
  } else {
    return new AliasImplementation();
  }
}

// implementation using data streams
class DataStreamImplementation {
  isUsingDataStreams() {
    return true;
  }
  getIndexTemplateFields(alias, patterns) {
    return {
      data_stream: {
        hidden: true
      },
      index_patterns: [alias]
    };
  }
  async createStream(opts) {
    return createDataStream(opts);
  }
}

// implementation using aliases and backing indices
class AliasImplementation {
  isUsingDataStreams() {
    return false;
  }
  getIndexTemplateFields(alias, patterns) {
    return {
      index_patterns: patterns,
      rollover_alias: alias
    };
  }
  async createStream(opts) {
    return createAliasStream(opts);
  }
}
async function createDataStream(opts) {
  const {
    logger,
    esClient,
    indexPatterns,
    totalFieldsLimit
  } = opts;
  logger.debug(`Creating data stream - ${indexPatterns.alias}`);

  // check if data stream exists
  let dataStream;
  try {
    const response = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.getDataStream({
      name: indexPatterns.alias,
      expand_wildcards: 'all'
    }), {
      logger
    });
    dataStream = response.data_streams[0];
  } catch (error) {
    if ((error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
      logger.error(`Error fetching data stream for ${indexPatterns.alias} - ${error.message}`);
      throw error;
    }
  }

  // if a data stream exists, update the underlying mapping
  if (dataStream) {
    const simulatedMapping = await simulateIndexMapping({
      logger,
      esClient,
      index: indexPatterns.alias
    });
    try {
      await (0, _create_concrete_write_index.updateIndexMappingsAndSettings)({
        logger,
        esClient,
        totalFieldsLimit,
        concreteIndices: [{
          alias: indexPatterns.alias,
          index: indexPatterns.alias,
          isWriteIndex: true
        }],
        simulatedMapping
      });
    } catch (err) {
      logger.error(`Failed to update mappings for data stream: ${indexPatterns.alias}, updating write index (${dataStream.indices[dataStream.indices.length - 1].index_name}) mappings instead`);
      try {
        await (0, _create_concrete_write_index.updateIndexMappingsAndSettings)({
          logger,
          esClient,
          totalFieldsLimit,
          concreteIndices: [{
            alias: indexPatterns.alias,
            index: dataStream.indices[dataStream.indices.length - 1].index_name,
            isWriteIndex: true
          }],
          simulatedMapping
        });
      } catch (innerError) {
        logger.error(`Failed to update mappings for write index of data stream: ${indexPatterns.alias}, rolling over instead`);
        await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.rollover({
          alias: indexPatterns.alias
        }), {
          logger
        });
      }
    }
  } else {
    try {
      await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.createDataStream({
        name: indexPatterns.alias
      }), {
        logger
      });
    } catch (error) {
      var _error$meta, _error$meta$body, _error$meta$body$erro;
      if ((error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.type) !== 'resource_already_exists_exception') {
        logger.error(`Error creating data stream ${indexPatterns.alias} - ${error.message}`);
        throw error;
      }
    }
  }
}
async function getValidConcreteIndices({
  indexPatterns,
  logger,
  esClient
}) {
  try {
    // Specify both the index pattern for the backing indices and their aliases
    // The alias prevents the request from finding other namespaces that could match the -* pattern
    const patterns = [indexPatterns.pattern];
    if (indexPatterns.reindexedPattern) {
      patterns.push(indexPatterns.reindexedPattern);
    }
    const response = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.getAlias({
      index: patterns,
      name: indexPatterns.alias
    }), {
      logger
    });
    const concreteIndices = Object.entries(response).flatMap(([index, {
      aliases
    }]) => Object.entries(aliases).map(([aliasName, aliasProperties]) => {
      var _aliasProperties$is_w;
      return {
        index,
        alias: aliasName,
        isWriteIndex: (_aliasProperties$is_w = aliasProperties.is_write_index) !== null && _aliasProperties$is_w !== void 0 ? _aliasProperties$is_w : false
      };
    }));
    logger.info(() => `Found ${concreteIndices.length} concrete indices for ${indexPatterns.name} - ${JSON.stringify(concreteIndices)}`);
    if (indexPatterns.validPrefixes) {
      const validConcreteIndices = [];
      for (const cIdx of concreteIndices) {
        if (!indexPatterns.validPrefixes.some(prefix => cIdx.index.startsWith(prefix))) {
          logger.warn(`Found unexpected concrete index name "${cIdx.index}" while expecting index with one of the following prefixes: [${indexPatterns.validPrefixes.join(',')}] Not updating mappings or settings for this index.`);
        } else {
          validConcreteIndices.push(cIdx);
        }
      }
      return validConcreteIndices;
    } else {
      return concreteIndices;
    }
  } catch (error) {
    // 404 is expected if no concrete write indices have been created
    if (error.statusCode !== 404) {
      logger.error(`Error fetching concrete indices for ${indexPatterns.pattern} pattern - ${error.message}`);
      throw error;
    } else {
      return [];
    }
  }
}
async function simulateIndexMapping({
  logger,
  esClient,
  index
}) {
  let simulatedIndexMapping;
  try {
    simulatedIndexMapping = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.simulateIndexTemplate({
      name: index
    }), {
      logger
    });
  } catch (err) {
    logger.error(`Ignored PUT mappings for ${index}; error generating simulated mappings: ${err.message}`);
    return;
  }
  const mapping = (0, _lodash.get)(simulatedIndexMapping, ['template', 'mappings']);
  if (mapping == null) {
    logger.error(`Ignored PUT mappings for ${index}; simulated mappings were empty`);
  }
  return mapping;
}
async function createAliasStream(opts) {
  const {
    logger,
    esClient,
    indexPatterns,
    totalFieldsLimit
  } = opts;
  logger.debug(`Creating concrete write index - ${indexPatterns.name}`);
  const validConcreteIndices = await getValidConcreteIndices({
    indexPatterns,
    esClient,
    logger
  });

  // if a concrete write index already exists, update the underlying mapping
  if (validConcreteIndices.length > 0) {
    const concreteWriteIndex = await (0, _create_concrete_write_index.findOrSetConcreteWriteIndex)({
      logger,
      esClient,
      concreteIndices: validConcreteIndices,
      alias: indexPatterns.alias
    });
    const nonWriteIndices = validConcreteIndices.filter(index => index.index !== concreteWriteIndex.index);
    const simulatedMapping = await simulateIndexMapping({
      logger,
      esClient,
      index: concreteWriteIndex.index
    });
    // Update the mappings for all indices other than the write index, if this fails, log an error but continue on
    try {
      await (0, _create_concrete_write_index.updateIndexMappingsAndSettings)({
        logger,
        esClient,
        totalFieldsLimit,
        concreteIndices: nonWriteIndices,
        simulatedMapping
      });
    } catch (err) {
      logger.error(`Failed to update mappings for concrete indices matching: ${indexPatterns.pattern}`);
    }

    // For the write index, try updating the mappings. If this fails, try rolling over. If rolling over fails,
    // throw an error.
    try {
      await (0, _create_concrete_write_index.updateIndexMappingsAndSettings)({
        logger,
        esClient,
        totalFieldsLimit,
        concreteIndices: [concreteWriteIndex],
        simulatedMapping
      });
    } catch (err) {
      logger.error(`Failed to update mappings for write index of alias: ${concreteWriteIndex.alias}, rolling over instead`);
      await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.rollover({
        alias: concreteWriteIndex.alias
      }), {
        logger
      });
    }
  } else {
    try {
      await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.create({
        index: indexPatterns.name,
        aliases: {
          [indexPatterns.alias]: {
            is_write_index: true
          }
        }
      }), {
        logger
      });
    } catch (error) {
      var _error$meta2, _error$meta2$body, _error$meta2$body$err;
      logger.error(`Error creating concrete write index - ${error.message}`);
      // If the index already exists and it's the write index for the alias,
      // something else created it so suppress the error. If it's not the write
      // index, that's bad, throw an error.
      if ((error === null || error === void 0 ? void 0 : (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : (_error$meta2$body$err = _error$meta2$body.error) === null || _error$meta2$body$err === void 0 ? void 0 : _error$meta2$body$err.type) === 'resource_already_exists_exception') {
        var _existingIndices$inde, _existingIndices$inde2, _existingIndices$inde3;
        const existingIndices = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.get({
          index: indexPatterns.name
        }), {
          logger
        });
        if (!((_existingIndices$inde = existingIndices[indexPatterns.name]) !== null && _existingIndices$inde !== void 0 && (_existingIndices$inde2 = _existingIndices$inde.aliases) !== null && _existingIndices$inde2 !== void 0 && (_existingIndices$inde3 = _existingIndices$inde2[indexPatterns.alias]) !== null && _existingIndices$inde3 !== void 0 && _existingIndices$inde3.is_write_index)) {
          throw Error(`Attempted to create index: ${indexPatterns.name} as the write index for alias: ${indexPatterns.alias}, but the index already exists and is not the write index for the alias`);
        }
      } else {
        throw error;
      }
    }
  }
}