"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSearchService = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createSearchService = dataClient => {
  const esClient = dataClient.deps.clusterClient.asCurrentUser;
  const getMonitoredUsers = batchUsernames => {
    return esClient.search({
      index: dataClient.index,
      size: batchUsernames.length,
      query: {
        bool: {
          must: [{
            terms: {
              'user.name': batchUsernames
            }
          }]
        }
      }
    });
  };
  const searchUsernamesInIndex = async ({
    indexName,
    batchSize,
    searchAfter,
    query
  }) => {
    return esClient.search({
      index: indexName,
      size: batchSize,
      _source: ['user.name'],
      sort: [{
        'user.name': 'asc'
      }],
      search_after: searchAfter,
      query
    });
  };
  const getExistingUsersMap = async usernames => {
    const existingUserRes = await getMonitoredUsers(usernames);
    const existingUserMap = new Map();
    for (const hit of existingUserRes.hits.hits) {
      var _hit$_source, _hit$_source$user;
      const username = (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : (_hit$_source$user = _hit$_source.user) === null || _hit$_source$user === void 0 ? void 0 : _hit$_source$user.name;
      dataClient.log('debug', `Found existing user: ${username} with ID: ${hit._id}`);
      if (username) existingUserMap.set(username, hit._id);
    }
    return existingUserMap;
  };
  return {
    getMonitoredUsers,
    searchUsernamesInIndex,
    getExistingUsersMap
  };
};
exports.createSearchService = createSearchService;