"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.turnOffPolicyProtectionsIfNotSupported = void 0;
var _keys = require("@kbn/security-solution-features/keys");
var _lodash = require("lodash");
var _policy_config_helpers = require("../../../common/endpoint/models/policy_config_helpers");
var _policy = require("../../../common/endpoint/service/policy");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const turnOffPolicyProtectionsIfNotSupported = async (esClient, fleetServices, productFeaturesService, logger) => {
  const log = logger.get('endpoint', 'policyProtectionsComplianceChecks');
  const isProtectionUpdatesFeatureEnabled = productFeaturesService.isEnabled(_keys.ProductFeatureSecurityKey.endpointProtectionUpdates);
  const isPolicyProtectionsEnabled = productFeaturesService.isEnabled(_keys.ProductFeatureSecurityKey.endpointPolicyProtections);
  const isCustomNotificationEnabled = productFeaturesService.isEnabled(_keys.ProductFeatureSecurityKey.endpointCustomNotification);
  if (isPolicyProtectionsEnabled && isProtectionUpdatesFeatureEnabled && isCustomNotificationEnabled) {
    log.info('All relevant features are enabled. Nothing to do!');
    return;
  }
  const {
    packagePolicy,
    savedObjects,
    endpointPolicyKuery
  } = fleetServices;
  const internalSoClient = savedObjects.createInternalScopedSoClient({
    readonly: false
  });
  const updates = [];
  const messages = [];
  const perPage = 1000;
  let hasMoreData = true;
  let page = 1;
  while (hasMoreData) {
    const {
      items,
      total
    } = await packagePolicy.list(internalSoClient, {
      page,
      kuery: endpointPolicyKuery,
      perPage
    });
    hasMoreData = page * perPage < total;
    page++;
    for (const item of items) {
      const integrationPolicy = item;
      const policySettings = integrationPolicy.inputs[0].config.policy.value;
      const {
        message,
        isOnlyCollectingEvents
      } = (0, _policy_config_helpers.isPolicySetToEventCollectionOnly)(policySettings);
      const shouldDowngradeProtectionUpdates = !isProtectionUpdatesFeatureEnabled && policySettings.global_manifest_version !== 'latest';
      const shouldDowngradeCustomNotifications = !isCustomNotificationEnabled && (0, _policy_config_helpers.checkIfPopupMessagesContainCustomNotifications)(policySettings);
      if (!isPolicyProtectionsEnabled && !isOnlyCollectingEvents || shouldDowngradeProtectionUpdates || shouldDowngradeCustomNotifications) {
        if (!isPolicyProtectionsEnabled && !isOnlyCollectingEvents) {
          messages.push(`Policy [${integrationPolicy.id}][${integrationPolicy.name}] updated to disable protections. Trigger: [${message}]`);
        }
        if (shouldDowngradeProtectionUpdates) {
          messages.push(`Policy [${integrationPolicy.id}][${integrationPolicy.name}] updated to downgrade protection updates.`);
        }
        if (shouldDowngradeCustomNotifications) {
          messages.push(`Policy [${integrationPolicy.id}][${integrationPolicy.name}] updated to reset custom notifications.`);
        }
        integrationPolicy.inputs[0].config.policy.value = (0, _lodash.merge)({}, policySettings, !isPolicyProtectionsEnabled && !isOnlyCollectingEvents ? (0, _policy_config_helpers.ensureOnlyEventCollectionIsAllowed)(policySettings) : {}, shouldDowngradeProtectionUpdates ? {
          global_manifest_version: 'latest'
        } : {}, shouldDowngradeCustomNotifications ? (0, _policy_config_helpers.resetCustomNotifications)() : {});
        updates.push({
          ...(0, _policy.getPolicyDataForUpdate)(integrationPolicy),
          id: integrationPolicy.id
        });
      }
    }
  }
  if (updates.length > 0) {
    log.info(`Found ${updates.length} policies that need updates:\n${messages.join('\n')}`);
    const bulkUpdateResponse = await fleetServices.packagePolicy.bulkUpdate(internalSoClient, esClient, updates, {
      user: {
        username: 'elastic'
      }
    });
    log.debug(() => `Bulk update response:\n${JSON.stringify(bulkUpdateResponse, null, 2)}`);
    if (bulkUpdateResponse.failedPolicies.length > 0) {
      log.error(`Done. ${bulkUpdateResponse.failedPolicies.length} out of ${updates.length} failed to update:\n${JSON.stringify(bulkUpdateResponse.failedPolicies, null, 2)}`);
    } else {
      log.info('Done. All updates applied successfully');
    }
  } else {
    log.info(`Done. Checked ${page * perPage} policies and no updates needed`);
  }
};
exports.turnOffPolicyProtectionsIfNotSupported = turnOffPolicyProtectionsIfNotSupported;