"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createLayerDescriptor = createLayerDescriptor;
var _uuid = require("uuid");
var _i18n = require("@kbn/i18n");
var _apmDataView = require("@kbn/apm-data-view");
var _constants = require("../../../../../../common/constants");
var _get_agg_key = require("../../../../../../common/get_agg_key");
var _layer_select = require("./layer_select");
var _metric_select = require("./metric_select");
var _display_select = require("./display_select");
var _vector_style = require("../../../../styles/vector/vector_style");
var _ems_file_source = require("../../../../sources/ems_file_source");
var _es_geo_grid_source = require("../../../../sources/es_geo_grid_source");
var _vector_layer = require("../../../vector_layer");
var _heatmap_layer = require("../../../heatmap_layer");
var _vector_style_defaults = require("../../../../styles/vector/vector_style_defaults");
var _kibana_services = require("../../../../../kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultDynamicProperties = (0, _vector_style_defaults.getDefaultDynamicProperties)();
function createDynamicFillColorDescriptor(layer, field) {
  return {
    type: _constants.STYLE_TYPE.DYNAMIC,
    options: {
      ...defaultDynamicProperties[_constants.VECTOR_STYLES.FILL_COLOR].options,
      field,
      color: layer === _layer_select.OBSERVABILITY_LAYER_TYPE.APM_RUM_PERFORMANCE ? 'Green to Red' : 'Yellow to Red',
      type: _constants.COLOR_MAP_TYPE.ORDINAL
    }
  };
}
function createLayerLabel(layer, metric) {
  let layerName;
  if (layer === _layer_select.OBSERVABILITY_LAYER_TYPE.APM_RUM_PERFORMANCE) {
    layerName = _i18n.i18n.translate('xpack.maps.observability.apmRumPerformanceLayerName', {
      defaultMessage: 'Performance'
    });
  } else if (layer === _layer_select.OBSERVABILITY_LAYER_TYPE.APM_RUM_TRAFFIC) {
    layerName = _i18n.i18n.translate('xpack.maps.observability.apmRumTrafficLayerName', {
      defaultMessage: 'Traffic'
    });
  }
  let metricName;
  if (metric === _metric_select.OBSERVABILITY_METRIC_TYPE.TRANSACTION_DURATION) {
    metricName = _i18n.i18n.translate('xpack.maps.observability.durationMetricName', {
      defaultMessage: 'Duration'
    });
  } else if (metric === _metric_select.OBSERVABILITY_METRIC_TYPE.COUNT) {
    metricName = _i18n.i18n.translate('xpack.maps.observability.countMetricName', {
      defaultMessage: 'Total'
    });
  } else if (metric === _metric_select.OBSERVABILITY_METRIC_TYPE.UNIQUE_COUNT) {
    metricName = _i18n.i18n.translate('xpack.maps.observability.uniqueCountMetricName', {
      defaultMessage: 'Unique count'
    });
  }
  return `[${layerName}] ${metricName}`;
}
function createAggDescriptor(metric) {
  if (metric === _metric_select.OBSERVABILITY_METRIC_TYPE.TRANSACTION_DURATION) {
    return {
      type: _constants.AGG_TYPE.AVG,
      field: 'transaction.duration.us'
    };
  } else if (metric === _metric_select.OBSERVABILITY_METRIC_TYPE.UNIQUE_COUNT) {
    return {
      type: _constants.AGG_TYPE.UNIQUE_COUNT,
      field: 'transaction.id'
    };
  } else {
    return {
      type: _constants.AGG_TYPE.COUNT
    };
  }
}

// All APM indices match APM index pattern. Need to apply query to filter to specific document types
// https://www.elastic.co/guide/en/kibana/current/apm-settings-kb.html
function createAmpSourceQuery(layer) {
  // APM transaction documents
  let query;
  if (layer === _layer_select.OBSERVABILITY_LAYER_TYPE.APM_RUM_PERFORMANCE || layer === _layer_select.OBSERVABILITY_LAYER_TYPE.APM_RUM_TRAFFIC) {
    query = 'processor.event:"transaction"';
  }
  return query ? {
    language: 'kuery',
    query
  } : undefined;
}
function getGeoGridRequestType(display) {
  if (display === _display_select.DISPLAY.HEATMAP) {
    return _constants.RENDER_AS.HEATMAP;
  }
  if (display === _display_select.DISPLAY.GRIDS) {
    return _constants.RENDER_AS.GRID;
  }
  return _constants.RENDER_AS.POINT;
}
function createLayerDescriptor({
  layer,
  metric,
  display
}) {
  if (!layer || !metric || !display) {
    return null;
  }
  const apmSourceQuery = createAmpSourceQuery(layer);
  const label = createLayerLabel(layer, metric);
  const metricsDescriptor = createAggDescriptor(metric);
  const apmDataViewId = (0, _apmDataView.getStaticDataViewId)((0, _kibana_services.getSpaceId)());
  if (display === _display_select.DISPLAY.CHOROPLETH) {
    const joinId = (0, _uuid.v4)();
    const joinKey = (0, _get_agg_key.getJoinAggKey)({
      aggType: metricsDescriptor.type,
      aggFieldName: 'field' in metricsDescriptor ? metricsDescriptor.field : '',
      rightSourceId: joinId
    });
    return _vector_layer.GeoJsonVectorLayer.createDescriptor({
      label,
      joins: [{
        leftField: 'iso2',
        right: {
          type: _constants.SOURCE_TYPES.ES_TERM_SOURCE,
          id: joinId,
          indexPatternId: apmDataViewId,
          term: 'client.geo.country_iso_code',
          metrics: [metricsDescriptor],
          whereQuery: apmSourceQuery,
          applyGlobalQuery: true,
          applyGlobalTime: true,
          applyForceRefresh: true
        }
      }],
      sourceDescriptor: _ems_file_source.EMSFileSource.createDescriptor({
        id: _constants.emsWorldLayerId,
        tooltipProperties: ['name', 'iso2']
      }),
      style: _vector_style.VectorStyle.createDescriptor({
        [_constants.VECTOR_STYLES.FILL_COLOR]: createDynamicFillColorDescriptor(layer, {
          name: joinKey,
          origin: _constants.FIELD_ORIGIN.JOIN
        }),
        [_constants.VECTOR_STYLES.LINE_COLOR]: {
          type: _constants.STYLE_TYPE.STATIC,
          options: {
            color: '#3d3d3d'
          }
        }
      })
    });
  }
  const geoGridSourceDescriptor = _es_geo_grid_source.ESGeoGridSource.createDescriptor({
    indexPatternId: apmDataViewId,
    geoField: 'client.geo.location',
    metrics: [metricsDescriptor],
    requestType: getGeoGridRequestType(display),
    resolution: _constants.GRID_RESOLUTION.MOST_FINE
  });
  if (display === _display_select.DISPLAY.HEATMAP) {
    return _heatmap_layer.HeatmapLayer.createDescriptor({
      label,
      query: apmSourceQuery,
      sourceDescriptor: geoGridSourceDescriptor
    });
  }
  const metricSourceKey = (0, _get_agg_key.getSourceAggKey)({
    aggType: metricsDescriptor.type,
    aggFieldName: 'field' in metricsDescriptor ? metricsDescriptor.field : undefined
  });
  const metricStyleField = {
    name: metricSourceKey,
    origin: _constants.FIELD_ORIGIN.SOURCE
  };
  const styleProperties = {
    [_constants.VECTOR_STYLES.FILL_COLOR]: createDynamicFillColorDescriptor(layer, metricStyleField),
    [_constants.VECTOR_STYLES.ICON_SIZE]: {
      type: _constants.STYLE_TYPE.DYNAMIC,
      options: {
        ...defaultDynamicProperties[_constants.VECTOR_STYLES.ICON_SIZE].options,
        field: metricStyleField
      }
    },
    [_constants.VECTOR_STYLES.LINE_COLOR]: {
      type: _constants.STYLE_TYPE.STATIC,
      options: {
        color: '#3d3d3d'
      }
    }
  };
  return _vector_layer.GeoJsonVectorLayer.createDescriptor({
    label,
    query: apmSourceQuery,
    sourceDescriptor: geoGridSourceDescriptor,
    style: _vector_style.VectorStyle.createDescriptor(styleProperties)
  });
}