"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BlendedVectorLayer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _i18n = require("@kbn/i18n");
var _geojson_vector_layer = require("../geojson_vector_layer");
var _vector_style = require("../../../styles/vector/vector_style");
var _vector_style_defaults = require("../../../styles/vector/vector_style_defaults");
var _constants = require("../../../../../common/constants");
var _es_geo_grid_source = require("../../../sources/es_geo_grid_source/es_geo_grid_source");
var _can_skip_fetch = require("../../../util/can_skip_fetch");
var _data_request = require("../../../util/data_request");
var _es_source = require("../../../sources/es_source/es_source");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ACTIVE_COUNT_DATA_ID = 'ACTIVE_COUNT_DATA_ID';
function getAggType(dynamicProperty) {
  return dynamicProperty.isOrdinal() ? _constants.AGG_TYPE.AVG : _constants.AGG_TYPE.TERMS;
}
function getClusterSource(documentSource, documentStyle) {
  const clusterSourceDescriptor = _es_geo_grid_source.ESGeoGridSource.createDescriptor({
    indexPatternId: documentSource.getIndexPatternId(),
    geoField: documentSource.getGeoFieldName(),
    requestType: _constants.RENDER_AS.POINT,
    resolution: _constants.GRID_RESOLUTION.COARSE
  });
  clusterSourceDescriptor.applyGlobalQuery = documentSource.getApplyGlobalQuery();
  clusterSourceDescriptor.applyGlobalTime = documentSource.getApplyGlobalTime();
  clusterSourceDescriptor.applyForceRefresh = documentSource.getApplyForceRefresh();
  clusterSourceDescriptor.metrics = [{
    type: _constants.AGG_TYPE.COUNT
  }, ...documentStyle.getDynamicPropertiesArray().map(dynamicProperty => {
    return {
      type: getAggType(dynamicProperty),
      field: dynamicProperty.getFieldName()
    };
  })];
  clusterSourceDescriptor.id = documentSource.getId();
  return new _es_geo_grid_source.ESGeoGridSource(clusterSourceDescriptor);
}
function getClusterStyleDescriptor(documentStyle, clusterSource) {
  const defaultDynamicProperties = (0, _vector_style_defaults.getDefaultDynamicProperties)();
  const clusterStyleDescriptor = {
    type: _constants.LAYER_STYLE_TYPE.VECTOR,
    properties: {
      [_constants.VECTOR_STYLES.LABEL_TEXT]: {
        type: _constants.STYLE_TYPE.DYNAMIC,
        options: {
          ...defaultDynamicProperties[_constants.VECTOR_STYLES.LABEL_TEXT].options,
          field: {
            name: _constants.COUNT_PROP_NAME,
            origin: _constants.FIELD_ORIGIN.SOURCE
          }
        }
      },
      [_constants.VECTOR_STYLES.ICON_SIZE]: {
        type: _constants.STYLE_TYPE.DYNAMIC,
        options: {
          ...defaultDynamicProperties[_constants.VECTOR_STYLES.ICON_SIZE].options,
          field: {
            name: _constants.COUNT_PROP_NAME,
            origin: _constants.FIELD_ORIGIN.SOURCE
          }
        }
      }
    },
    isTimeAware: true
  };
  documentStyle.getAllStyleProperties().forEach(styleProperty => {
    const styleName = styleProperty.getStyleName();
    if ([_constants.VECTOR_STYLES.LABEL_TEXT, _constants.VECTOR_STYLES.ICON_SIZE].includes(styleName) && (!styleProperty.isDynamic() || !styleProperty.isComplete())) {
      // Do not migrate static label and icon size properties to provide unique cluster styling out of the box
      return;
    }
    if (styleName === _constants.VECTOR_STYLES.SYMBOLIZE_AS || styleName === _constants.VECTOR_STYLES.LABEL_BORDER_SIZE) {
      // copy none static/dynamic styles to cluster style
      clusterStyleDescriptor.properties[styleName] = {
        // @ts-expect-error
        options: {
          ...styleProperty.getOptions()
        }
      };
    } else if (styleProperty.isDynamic()) {
      // copy dynamic styles to cluster style
      const options = styleProperty.getOptions();
      const field = options && options.field && options.field.name ? {
        ...options.field,
        name: clusterSource.getAggKey(getAggType(styleProperty), options.field.name)
      } : undefined;
      clusterStyleDescriptor.properties[styleName] = {
        type: _constants.STYLE_TYPE.DYNAMIC,
        // @ts-expect-error upgrade typescript v5.1.6
        options: {
          ...options,
          field
        }
      };
    } else {
      // copy static styles to cluster style
      clusterStyleDescriptor.properties[styleName] = {
        type: _constants.STYLE_TYPE.STATIC,
        // @ts-expect-error upgrade typescript v5.1.6
        options: {
          ...styleProperty.getOptions()
        }
      };
    }
  });
  return clusterStyleDescriptor;
}
class BlendedVectorLayer extends _geojson_vector_layer.GeoJsonVectorLayer {
  static createDescriptor(options, mapColors) {
    const layerDescriptor = _geojson_vector_layer.GeoJsonVectorLayer.createDescriptor(options, mapColors);
    layerDescriptor.type = _constants.LAYER_TYPE.BLENDED_VECTOR;
    return layerDescriptor;
  }
  constructor(options) {
    super({
      ...options,
      joins: []
    });
    (0, _defineProperty2.default)(this, "_isClustered", void 0);
    (0, _defineProperty2.default)(this, "_clusterSource", void 0);
    (0, _defineProperty2.default)(this, "_clusterStyle", void 0);
    (0, _defineProperty2.default)(this, "_documentSource", void 0);
    (0, _defineProperty2.default)(this, "_documentStyle", void 0);
    (0, _defineProperty2.default)(this, "getSource", () => {
      return this._isClustered ? this._clusterSource : this._documentSource;
    });
    this._documentSource = this._source; // VectorLayer constructor sets _source as document source
    this._documentStyle = this._style; // VectorLayer constructor sets _style as document source

    this._clusterSource = getClusterSource(this._documentSource, this._documentStyle);
    const clusterStyleDescriptor = getClusterStyleDescriptor(this._documentStyle, this._clusterSource);
    this._clusterStyle = new _vector_style.VectorStyle(clusterStyleDescriptor, this._clusterSource, this, options.customIcons, options.chartsPaletteServiceGetColor);
    let isClustered = false;
    const countDataRequest = this.getDataRequest(ACTIVE_COUNT_DATA_ID);
    if (countDataRequest) {
      const requestData = countDataRequest.getData();
      if (requestData && requestData.isSyncClustered) {
        isClustered = true;
      }
    }
    this._isClustered = isClustered;
  }
  async getDisplayName(source) {
    const displayName = await super.getDisplayName(source);
    return this._isClustered ? _i18n.i18n.translate('xpack.maps.blendedVectorLayer.clusteredLayerName', {
      defaultMessage: 'Clustered {displayName}',
      values: {
        displayName
      }
    }) : displayName;
  }
  getJoins() {
    return [];
  }
  hasJoins() {
    return false;
  }
  async cloneDescriptor() {
    const clones = await super.cloneDescriptor();
    if (clones.length === 0) {
      return [];
    }
    const clonedDescriptor = clones[0];

    // Use super getDisplayName instead of instance getDisplayName to avoid getting 'Clustered Clone of Clustered'
    const displayName = await super.getDisplayName();
    clonedDescriptor.label = `Clone of ${displayName}`;

    // sourceDescriptor must be document source descriptor
    clonedDescriptor.sourceDescriptor = this._documentSource.cloneDescriptor();
    return [clonedDescriptor];
  }
  getSourceForEditing() {
    // Layer is based on this._documentSource
    // this._clusterSource is a derived source for rendering only.
    // Regardless of this._activeSource, this._documentSource should always be displayed in the editor
    return this._documentSource;
  }
  getCurrentStyle() {
    return this._isClustered ? this._clusterStyle : this._documentStyle;
  }
  getStyleForEditing() {
    return this._documentStyle;
  }
  async syncData(syncContext) {
    const dataRequestId = ACTIVE_COUNT_DATA_ID;
    const requestToken = Symbol(`layer-active-count:${this.getId()}`);
    const requestMeta = await this._getVectorSourceRequestMeta(syncContext.isForceRefresh, syncContext.dataFilters, this.getSource(), this.getCurrentStyle(), syncContext.isFeatureEditorOpenForLayer);
    const source = this.getSource();
    const canSkipSourceFetch = await (0, _can_skip_fetch.canSkipSourceUpdate)({
      source,
      prevDataRequest: this.getDataRequest(dataRequestId),
      nextRequestMeta: requestMeta,
      extentAware: source.isFilterByMapBounds(),
      getUpdateDueToTimeslice: timeslice => {
        return this._getUpdateDueToTimesliceFromSourceRequestMeta(source, timeslice);
      }
    });
    let activeSource;
    let activeStyle;
    if (canSkipSourceFetch) {
      // Even when source fetch is skipped, need to call super._syncData to sync StyleMeta and formatters
      if (this._isClustered) {
        activeSource = this._clusterSource;
        activeStyle = this._clusterStyle;
      } else {
        activeSource = this._documentSource;
        activeStyle = this._documentStyle;
      }
    } else {
      let isSyncClustered;
      try {
        syncContext.startLoading(dataRequestId, requestToken, requestMeta);
        const warnings = [];
        isSyncClustered = !(await this._documentSource.canLoadAllDocuments(await this.getDisplayName(this._documentSource), requestMeta, syncContext.registerCancelCallback.bind(null, requestToken), syncContext.inspectorAdapters, warning => {
          warnings.push(warning);
        }));
        syncContext.stopLoading(dataRequestId, requestToken, {
          isSyncClustered
        }, {
          ...requestMeta,
          warnings
        });
      } catch (error) {
        if (!(error instanceof _data_request.DataRequestAbortError) || !(0, _es_source.isSearchSourceAbortError)(error)) {
          syncContext.onLoadError(dataRequestId, requestToken, error);
        }
        return;
      }
      if (isSyncClustered) {
        this._isClustered = true;
        activeSource = this._clusterSource;
        activeStyle = this._clusterStyle;
      } else {
        this._isClustered = false;
        activeSource = this._documentSource;
        activeStyle = this._documentStyle;
      }
    }
    super._syncData(syncContext, activeSource, activeStyle);
  }
  async getLicensedFeatures() {
    return [...(await this._clusterSource.getLicensedFeatures()), ...(await this._documentSource.getLicensedFeatures())];
  }
}
exports.BlendedVectorLayer = BlendedVectorLayer;