"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLensVisualizationsGetAPIRoute = void 0;
var _boom = require("@hapi/boom");
var _constants = require("../../../../common/constants");
var _schema = require("./schema");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerLensVisualizationsGetAPIRoute = (router, {
  contentManagement
}) => {
  const getRoute = router.get({
    path: `${_constants.LENS_VIS_API_PATH}/{id}`,
    access: _constants.LENS_API_ACCESS,
    enableQueryVersion: true,
    summary: 'Get Lens visualization',
    description: 'Get a Lens visualization from id.',
    options: {
      tags: ['oas-tag:Lens'],
      availability: {
        stability: 'experimental'
      }
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on Content Client for authorization'
      }
    }
  });
  getRoute.addVersion({
    version: _constants.LENS_API_VERSION,
    validate: {
      request: {
        params: _schema.lensGetRequestParamsSchema
      },
      response: {
        200: {
          body: () => _schema.lensGetResponseBodySchema,
          description: 'Ok'
        },
        400: {
          description: 'Malformed request'
        },
        401: {
          description: 'Unauthorized'
        },
        403: {
          description: 'Forbidden'
        },
        404: {
          description: 'Resource not found'
        },
        500: {
          description: 'Internal Server Error'
        }
      }
    }
  }, async (ctx, req, res) => {
    // TODO fix IContentClient to type this client based on the actual
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_constants.LENS_CONTENT_TYPE);
    try {
      const {
        result
      } = await client.get(req.params.id);
      if (result.item.error) {
        throw result.item.error;
      }
      const body = (0, _utils.getLensResponseItem)(result.item);
      return res.ok({
        body: {
          ...body,
          meta: {
            ...body.meta,
            ...result.meta
          }
        }
      });
    } catch (error) {
      if ((0, _boom.isBoom)(error)) {
        if (error.output.statusCode === 404) {
          return res.notFound({
            body: {
              message: `A Lens visualization with id [${req.params.id}] was not found.`
            }
          });
        }
        if (error.output.statusCode === 403) {
          return res.forbidden();
        }
      }
      return (0, _boom.boomify)(error); // forward unknown error
    }
  });
};
exports.registerLensVisualizationsGetAPIRoute = registerLensVisualizationsGetAPIRoute;