"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDefendInsight = createDefendInsight;
exports.getAssistantTool = getAssistantTool;
exports.getAssistantToolParams = getAssistantToolParams;
exports.handleGraphError = void 0;
exports.handleToolError = handleToolError;
exports.invokeDefendInsightsGraph = void 0;
exports.isDefendInsightsPolicyResponseFailureEnabled = isDefendInsightsPolicyResponseFailureEnabled;
exports.throwIfErrorCountsExceeded = exports.runExternalCallbacks = void 0;
exports.updateDefendInsightLastViewedAt = updateDefendInsightLastViewedAt;
exports.updateDefendInsights = updateDefendInsights;
exports.updateDefendInsightsLastViewedAt = updateDefendInsightsLastViewedAt;
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/langchain/server");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _app_context = require("../../services/app_context");
var _prompts = require("../../lib/defend_insights/graphs/default_defend_insights_graph/prompts");
var _event_based_telemetry = require("../../lib/telemetry/event_based_telemetry");
var _default_defend_insights_graph = require("../../lib/defend_insights/graphs/default_defend_insights_graph");
var _constants = require("../../lib/defend_insights/graphs/default_defend_insights_graph/constants");
var _helpers = require("../helpers");
var _utils = require("../utils");
var _translations = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const KB_REQUIRED_TYPES = new Set([_elasticAssistantCommon.DefendInsightType.Enum.policy_response_failure]);
function addGenerationInterval(generationIntervals, generationInterval) {
  const newGenerationIntervals = [generationInterval, ...generationIntervals];
  const MAX_GENERATION_INTERVALS = 5;
  if (newGenerationIntervals.length > MAX_GENERATION_INTERVALS) {
    return newGenerationIntervals.slice(0, MAX_GENERATION_INTERVALS); // Return the first MAX_GENERATION_INTERVALS items
  }
  return newGenerationIntervals;
}
function isDefendInsightsPolicyResponseFailureEnabled({
  request,
  logger,
  assistantContext
}) {
  const pluginName = (0, _helpers.getPluginNameFromRequest)({
    request,
    logger,
    defaultPluginName: _helpers.DEFAULT_PLUGIN_NAME
  });
  return assistantContext.getRegisteredFeatures(pluginName).defendInsightsPolicyResponseFailure;
}
function getAssistantTool(getRegisteredTools, pluginName) {
  const assistantTools = getRegisteredTools(pluginName);
  return assistantTools.find(tool => tool.id === _elasticAssistantCommon.DEFEND_INSIGHTS_ID);
}
function getAssistantToolParams({
  endpointIds,
  insightType,
  actionsClient,
  anonymizationFields,
  apiConfig,
  esClient,
  connectorTimeout,
  langChainTimeout,
  langSmithProject,
  langSmithApiKey,
  logger,
  contentReferencesStore,
  latestReplacements,
  onNewReplacements,
  request
}) {
  const traceOptions = {
    projectName: langSmithProject,
    tracers: [...(0, _langsmith.getLangSmithTracer)({
      apiKey: langSmithApiKey,
      projectName: langSmithProject,
      logger
    })]
  };
  const llm = new _server.ActionsClientLlm({
    actionsClient,
    connectorId: apiConfig.connectorId,
    llmType: (0, _utils.getLlmType)(apiConfig.actionTypeId),
    logger,
    temperature: 0,
    // zero temperature because we want structured JSON output
    timeout: connectorTimeout,
    traceOptions,
    telemetryMetadata: {
      pluginId: 'security_defend_insights'
    }
  });
  return {
    endpointIds,
    insightType,
    anonymizationFields,
    esClient,
    replacements: latestReplacements,
    langChainTimeout,
    llm,
    logger,
    contentReferencesStore,
    onNewReplacements,
    request,
    modelExists: false,
    isEnabledKnowledgeBase: false
  };
}
async function handleToolError({
  apiConfig,
  defendInsightId,
  authenticatedUser,
  dataClient,
  err,
  latestReplacements,
  logger,
  telemetry
}) {
  try {
    logger.error(err);
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    const currentInsight = await dataClient.getDefendInsight({
      id: defendInsightId,
      authenticatedUser
    });
    if (currentInsight === null || (currentInsight === null || currentInsight === void 0 ? void 0 : currentInsight.status) === _elasticAssistantCommon.DefendInsightStatus.Enum.canceled) {
      return;
    }
    await dataClient.updateDefendInsight({
      defendInsightUpdateProps: {
        insights: [],
        status: _elasticAssistantCommon.DefendInsightStatus.Enum.failed,
        id: defendInsightId,
        replacements: latestReplacements,
        backingIndex: currentInsight.backingIndex,
        failureReason: error.message
      },
      authenticatedUser
    });
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: error.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  } catch (updateErr) {
    const updateError = (0, _securitysolutionEsUtils.transformError)(updateErr);
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: updateError.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  }
}
async function createDefendInsight(endpointIds, insightType, dataClient, authenticatedUser, apiConfig) {
  const currentInsight = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.createDefendInsight({
    defendInsightCreate: {
      endpointIds,
      insightType,
      apiConfig,
      insights: [],
      status: _elasticAssistantCommon.DefendInsightStatus.Enum.running
    },
    authenticatedUser
  }));
  if (!currentInsight) {
    throw new Error(`failed to create Defend insight for connectorId: ${apiConfig.connectorId}`);
  }
  return {
    defendInsightId: currentInsight.id,
    currentInsight
  };
}
const extractInsightsForTelemetryReporting = (insightType, insights) => {
  switch (insightType) {
    case _elasticAssistantCommon.DefendInsightType.Enum.incompatible_antivirus:
      return insights.map(insight => insight.group);
    case _elasticAssistantCommon.DefendInsightType.Enum.policy_response_failure:
      return insights.map(insight => insight.group);
    default:
      return [];
  }
};
async function updateDefendInsights({
  anonymizedEvents,
  apiConfig,
  defendInsightId,
  insights,
  authenticatedUser,
  dataClient,
  latestReplacements,
  logger,
  startTime,
  telemetry,
  insightType
}) {
  try {
    var _updateProps$insights, _updateProps$insights2;
    const currentInsight = await dataClient.getDefendInsight({
      id: defendInsightId,
      authenticatedUser
    });
    if (currentInsight === null || (currentInsight === null || currentInsight === void 0 ? void 0 : currentInsight.status) === _elasticAssistantCommon.DefendInsightStatus.Enum.canceled) {
      return;
    }
    const endTime = (0, _moment.default)();
    const durationMs = endTime.diff(startTime);
    const eventsContextCount = anonymizedEvents.length;
    const updateProps = {
      eventsContextCount,
      insights: insights !== null && insights !== void 0 ? insights : undefined,
      status: _elasticAssistantCommon.DefendInsightStatus.Enum.succeeded,
      ...(!eventsContextCount || !insights ? {} : {
        generationIntervals: addGenerationInterval(currentInsight.generationIntervals, {
          durationMs,
          date: new Date().toISOString()
        })
      }),
      id: defendInsightId,
      replacements: latestReplacements,
      backingIndex: currentInsight.backingIndex
    };
    await dataClient.updateDefendInsight({
      defendInsightUpdateProps: updateProps,
      authenticatedUser
    });
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_SUCCESS_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      eventsContextCount: updateProps.eventsContextCount,
      insightsGenerated: (_updateProps$insights = (_updateProps$insights2 = updateProps.insights) === null || _updateProps$insights2 === void 0 ? void 0 : _updateProps$insights2.length) !== null && _updateProps$insights !== void 0 ? _updateProps$insights : 0,
      insightsDetails: extractInsightsForTelemetryReporting(insightType, updateProps.insights || []),
      durationMs,
      model: apiConfig.model,
      provider: apiConfig.provider,
      insightType
    });
  } catch (updateErr) {
    logger.error(updateErr);
    const updateError = (0, _securitysolutionEsUtils.transformError)(updateErr);
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: updateError.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  }
}
async function updateDefendInsightsLastViewedAt({
  defendInsights,
  authenticatedUser,
  dataClient
}) {
  if (!defendInsights.length) {
    return [];
  }
  const defendInsightsUpdateProps = defendInsights.map(insight => {
    return {
      id: insight.id,
      lastViewedAt: new Date().toISOString(),
      backingIndex: insight.backingIndex
    };
  });
  return dataClient.updateDefendInsights({
    defendInsightsUpdateProps,
    authenticatedUser
  });
}
async function updateDefendInsightLastViewedAt({
  defendInsights,
  authenticatedUser,
  dataClient
}) {
  return (await updateDefendInsightsLastViewedAt({
    defendInsights,
    authenticatedUser,
    dataClient
  }))[0];
}
const invokeDefendInsightsGraph = async ({
  insightType,
  endpointIds,
  actionsClient,
  anonymizationFields,
  apiConfig,
  connectorTimeout,
  esClient,
  langSmithProject,
  langSmithApiKey,
  latestReplacements,
  logger,
  onNewReplacements,
  size,
  start,
  end,
  savedObjectsClient,
  kbDataClient
}) => {
  var _traceOptions$tracers;
  if (KB_REQUIRED_TYPES.has(insightType)) {
    await waitForKB(kbDataClient);
  }
  const llmType = (0, _utils.getLlmType)(apiConfig.actionTypeId);
  const model = apiConfig.model;
  const tags = [_elasticAssistantCommon.DEFEND_INSIGHTS_ID, llmType, model].flatMap(tag => tag !== null && tag !== void 0 ? tag : []);
  const traceOptions = {
    projectName: langSmithProject,
    tracers: [...(0, _langsmith.getLangSmithTracer)({
      apiKey: langSmithApiKey,
      projectName: langSmithProject,
      logger
    })]
  };
  const llm = new _server.ActionsClientLlm({
    actionsClient,
    connectorId: apiConfig.connectorId,
    llmType,
    logger,
    temperature: 0,
    timeout: connectorTimeout,
    traceOptions,
    telemetryMetadata: {
      pluginId: 'security_defend_insights'
    }
  });
  if (llm == null) {
    throw new Error('LLM is required for Defend insights');
  }
  const defendInsightsPrompts = await (0, _prompts.getDefendInsightsPrompt)({
    type: insightType,
    actionsClient,
    connectorId: apiConfig.connectorId,
    model,
    provider: llmType,
    savedObjectsClient
  });
  const graph = (0, _default_defend_insights_graph.getDefaultDefendInsightsGraph)({
    insightType,
    endpointIds,
    anonymizationFields,
    esClient,
    kbDataClient,
    llm,
    logger,
    onNewReplacements,
    prompts: defendInsightsPrompts,
    replacements: latestReplacements,
    size,
    start,
    end
  });
  logger === null || logger === void 0 ? void 0 : logger.debug(() => 'invokeDefendInsightsGraph: invoking the Defend insights graph');
  const result = await graph.invoke({}, {
    callbacks: [...((_traceOptions$tracers = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tracers) !== null && _traceOptions$tracers !== void 0 ? _traceOptions$tracers : [])],
    runName: _constants.DEFEND_INSIGHTS_GRAPH_RUN_NAME,
    tags
  });
  const {
    insights,
    anonymizedDocuments: anonymizedEvents,
    errors,
    generationAttempts,
    hallucinationFailures,
    maxGenerationAttempts,
    maxHallucinationFailures
  } = result;
  throwIfErrorCountsExceeded({
    errors,
    generationAttempts,
    hallucinationFailures,
    logger,
    maxGenerationAttempts,
    maxHallucinationFailures
  });
  return {
    anonymizedEvents,
    insights
  };
};
exports.invokeDefendInsightsGraph = invokeDefendInsightsGraph;
const handleGraphError = async ({
  apiConfig,
  defendInsightId,
  authenticatedUser,
  dataClient,
  err,
  latestReplacements,
  logger,
  telemetry
}) => {
  try {
    logger.error(err);
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    const currentInsight = await dataClient.getDefendInsight({
      id: defendInsightId,
      authenticatedUser
    });
    if (currentInsight === null || (currentInsight === null || currentInsight === void 0 ? void 0 : currentInsight.status) === 'canceled') {
      return;
    }
    await dataClient.updateDefendInsight({
      defendInsightUpdateProps: {
        insights: [],
        status: _elasticAssistantCommon.DefendInsightStatus.Enum.failed,
        id: defendInsightId,
        replacements: latestReplacements,
        backingIndex: currentInsight.backingIndex,
        failureReason: error.message
      },
      authenticatedUser
    });
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: error.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  } catch (updateErr) {
    const updateError = (0, _securitysolutionEsUtils.transformError)(updateErr);
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: updateError.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  }
};
exports.handleGraphError = handleGraphError;
const runExternalCallbacks = async (callback, ...args) => {
  const callbacks = _app_context.appContextService.getRegisteredCallbacks(callback);
  await Promise.all(callbacks.map(cb => Promise.resolve(cb(...args))));
};
exports.runExternalCallbacks = runExternalCallbacks;
const throwIfErrorCountsExceeded = ({
  errors,
  generationAttempts,
  hallucinationFailures,
  logger,
  maxGenerationAttempts,
  maxHallucinationFailures
}) => {
  if (hallucinationFailures >= maxHallucinationFailures) {
    const hallucinationFailuresError = `${(0, _translations.MAX_HALLUCINATION_FAILURES)(hallucinationFailures)}\n${errors.join(',\n')}`;
    logger === null || logger === void 0 ? void 0 : logger.error(hallucinationFailuresError);
    throw new Error(hallucinationFailuresError);
  }
  if (generationAttempts >= maxGenerationAttempts) {
    const generationAttemptsError = `${(0, _translations.MAX_GENERATION_ATTEMPTS)(generationAttempts)}\n${errors.join(',\n')}`;
    logger === null || logger === void 0 ? void 0 : logger.error(generationAttemptsError);
    throw new Error(generationAttemptsError);
  }
};
exports.throwIfErrorCountsExceeded = throwIfErrorCountsExceeded;
async function waitForKB(kbDataClient) {
  if (!kbDataClient) {
    return Promise.resolve();
  }
  if (await (kbDataClient === null || kbDataClient === void 0 ? void 0 : kbDataClient.isDefendInsightsDocsLoaded())) {
    return Promise.resolve();
  }
  if (kbDataClient !== null && kbDataClient !== void 0 && kbDataClient.isSetupInProgress) {
    return new Promise((resolve, reject) => {
      const interval = 30000;
      const maxTimeout = 10 * 60 * 1000;
      const startTime = Date.now();
      const checkKBStatus = async () => {
        try {
          const elapsedTime = Date.now() - startTime;
          if (elapsedTime > maxTimeout) {
            reject(new Error(`Knowledge base setup timed out after ${maxTimeout / 1000} seconds`));
            return;
          }
          if (await kbDataClient.isDefendInsightsDocsLoaded()) {
            resolve();
          } else {
            setTimeout(checkKBStatus, interval);
          }
        } catch (error) {
          reject(error);
        }
      };
      void checkKBStatus();
    });
  }
  return Promise.resolve();
}