"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCompletedCards = void 0;
var _react = require("react");
var _kibana = require("../../../../common/lib/kibana");
var _use_stored_state = require("../../hooks/use_stored_state");
var _onboarding_context = require("../../onboarding_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This hook implements the logic for tracking which onboarding cards have been completed using Local Storage.
 */
const useCompletedCards = bodyConfig => {
  const {
    spaceId,
    telemetry
  } = (0, _onboarding_context.useOnboardingContext)();
  const services = (0, _kibana.useKibana)().services;

  // Use stored state to keep localStorage in sync, and a local state to avoid unnecessary re-renders.
  const [storedCompleteCardIds, setStoredCompleteCardIds] = (0, _use_stored_state.useStoredCompletedCardIds)(spaceId);
  const [completeCardIds, setCompleteCardIds] = (0, _react.useState)(storedCompleteCardIds);
  // Local state to store the checkCompleteResult for each card
  const [cardCheckCompleteResult, setCardsCompleteResult] = (0, _react.useState)({});
  const isCardComplete = (0, _react.useCallback)(cardId => completeCardIds.includes(cardId), [completeCardIds]);
  const setCardComplete = (0, _react.useCallback)((cardId, completed, options) => {
    // This state update has side effects, using a callback
    setCompleteCardIds(currentCompleteCards => {
      const isCurrentlyComplete = currentCompleteCards.includes(cardId);
      if (completed && !isCurrentlyComplete) {
        const newCompleteCardIds = [...currentCompleteCards, cardId];
        telemetry.reportCardComplete(cardId, options);
        setStoredCompleteCardIds(newCompleteCardIds); // Keep the stored state in sync with the local state
        return newCompleteCardIds;
      } else if (!completed && isCurrentlyComplete) {
        const newCompleteCardIds = currentCompleteCards.filter(id => id !== cardId);
        setStoredCompleteCardIds(newCompleteCardIds); // Keep the stored state in sync with the local state
        return newCompleteCardIds;
      }
      return currentCompleteCards; // No change
    });
  }, [setStoredCompleteCardIds, telemetry] // static dependencies, this function needs to be stable
  );
  const getCardCheckCompleteResult = (0, _react.useCallback)(cardId => cardCheckCompleteResult[cardId], [cardCheckCompleteResult]);

  // Internal: sets the checkCompleteResult for a specific card
  const setCardCheckCompleteResult = (0, _react.useCallback)((cardId, options) => {
    setCardsCompleteResult((currentCardCheckCompleteResult = {}) => ({
      ...currentCardCheckCompleteResult,
      [cardId]: options
    }));
  }, []);

  // Internal: stores all cards that have a checkComplete function in a flat array
  const cardsWithAutoCheck = (0, _react.useMemo)(() => bodyConfig.reduce((acc, group) => {
    acc.push(...group.cards.filter(card => card.checkComplete));
    return acc;
  }, []), [bodyConfig]);

  // Internal: sets the result of a checkComplete function
  const processCardCheckCompleteResult = (0, _react.useCallback)((cardId, checkCompleteResult) => {
    if (typeof checkCompleteResult === 'boolean') {
      setCardComplete(cardId, checkCompleteResult, {
        auto: true
      });
    } else {
      const {
        isComplete,
        ...result
      } = checkCompleteResult;
      setCardComplete(cardId, isComplete, {
        auto: true
      });
      setCardCheckCompleteResult(cardId, result);
    }
  }, [setCardComplete, setCardCheckCompleteResult]);
  const checkCardComplete = (0, _react.useCallback)(cardId => {
    const cardConfig = cardsWithAutoCheck.find(({
      id
    }) => id === cardId);
    if (cardConfig) {
      var _cardConfig$checkComp;
      (_cardConfig$checkComp = cardConfig.checkComplete) === null || _cardConfig$checkComp === void 0 ? void 0 : _cardConfig$checkComp.call(cardConfig, services).catch(err => {
        services.notifications.toasts.addError(err, {
          title: cardConfig.title
        });
        return {
          isComplete: false
        };
      }).then(checkCompleteResult => {
        processCardCheckCompleteResult(cardId, checkCompleteResult);
      });
    }
  }, [cardsWithAutoCheck, processCardCheckCompleteResult, services]);
  (0, _react.useEffect)(() => {
    // Initial auto-check for all body cards, it should run once per `bodyConfig` (topic) change.
    cardsWithAutoCheck.map(card => {
      var _card$checkComplete;
      return (_card$checkComplete = card.checkComplete) === null || _card$checkComplete === void 0 ? void 0 : _card$checkComplete.call(card, services).catch(err => {
        services.notifications.toasts.addError(err, {
          title: card.title
        });
        return {
          isComplete: false
        };
      }).then(checkCompleteResult => {
        processCardCheckCompleteResult(card.id, checkCompleteResult);
      });
    });
  }, [cardsWithAutoCheck, processCardCheckCompleteResult, services]);
  return {
    isCardComplete,
    setCardComplete,
    getCardCheckCompleteResult,
    checkCardComplete
  };
};
exports.useCompletedCards = useCompletedCards;