"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAlertCloseInfoModal = exports.hasAlertsInSuppressionWindow = void 0;
var _react = _interopRequireWildcard(require("react"));
var _i18nReact = require("@kbn/i18n-react");
var _eui = require("@elastic/eui");
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _constants = require("../../../common/constants");
var _kibana = require("../../common/lib/kibana");
var i18n = _interopRequireWildcard(require("./translations"));
var _api = require("../containers/detection_engine/alerts/api");
var _jsxFileName = "/opt/buildkite-agent/builds/bk-agent-prod-gcp-1763985831484149691/elastic/kibana-artifacts-snapshot/kibana/x-pack/solutions/security/plugins/security_solution/public/detections/hooks/use_alert_close_info_modal.tsx";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
const DO_NOT_SHOW_AGAIN_SETTING_KEY = 'securitySolution.alertCloseInfoModal.doNotShowAgain';
const restartSuppressionMessageComponent = /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
  id: "xpack.securitySolution.alert.closeInfoModal.restartSuppressionMessage",
  defaultMessage: "Some of the alerts being closed were created while a suppression window was active. If suppression remains active, any new, duplicate events will be grouped and suppressed. Each unique group will be associated with a new alert.",
  __self: void 0,
  __source: {
    fileName: _jsxFileName,
    lineNumber: 24,
    columnNumber: 3
  }
});
const continueSuppressionMessageComponent = /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
  id: "xpack.securitySolution.alert.closeInfoModal.continueSuppressionMessage",
  defaultMessage: "Some of the alerts being closed were created while a suppression window was active. If suppression remains active, duplicate events will continue to be grouped and suppressed, but new alerts won't be created for these groups.",
  __self: void 0,
  __source: {
    fileName: _jsxFileName,
    lineNumber: 31,
    columnNumber: 3
  }
});

/**
 * Given a query or a list of ids used to search through alerts, return true if any of the matching alerts
 * have a suppression window configured in their alert properties
 */
const hasAlertsInSuppressionWindow = async params => {
  let query;
  if (params.ids && params.ids.length > 0) {
    query = {
      bool: {
        filter: [{
          ids: {
            values: params.ids
          }
        }]
      }
    };
  } else if (params.query) {
    query = JSON.parse(params.query);
  } else {
    throw new Error('either query or a non empty list of alert ids must be defined');
  }
  const boolFilters = (0, _lodash.get)(query, 'bool.filter', []);
  boolFilters.push({
    exists: {
      field: 'kibana.alert.rule.parameters.alert_suppression.duration.value'
    }
  });
  (0, _saferLodashSet.set)(query, 'bool.filter', boolFilters);
  const abortCtrl = new AbortController();
  const results = await (0, _api.fetchQueryAlerts)({
    query: {
      query,
      size: 0
    },
    signal: abortCtrl.signal
  });
  return results.hits.total.value > 0;
};
exports.hasAlertsInSuppressionWindow = hasAlertsInSuppressionWindow;
const AlertCloseConfirmationModal = ({
  onConfirmationResult
}) => {
  const {
    storage
  } = (0, _kibana.useKibana)().services;
  const [doNotShowAgain, setDoNotShowAgain] = (0, _react.useState)(false);
  const modalTitleId = (0, _eui.useGeneratedHtmlId)();
  const doNotShowAgainCheckboxId = (0, _eui.useGeneratedHtmlId)();
  const [currentSettingValue] = (0, _kibana.useUiSetting$)(_constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING);
  const onDoNotShowAgainTicked = (0, _react.useCallback)(() => {
    setDoNotShowAgain(!doNotShowAgain);
  }, [doNotShowAgain]);
  const onConfirm = (0, _react.useCallback)(() => {
    onConfirmationResult(true);
    storage.set(DO_NOT_SHOW_AGAIN_SETTING_KEY, doNotShowAgain);
  }, [onConfirmationResult, storage, doNotShowAgain]);
  const onCancel = (0, _react.useCallback)(() => onConfirmationResult(false), [onConfirmationResult]);
  const {
    title,
    message
  } = currentSettingValue === _constants.SUPPRESSION_BEHAVIOR_ON_ALERT_CLOSURE_SETTING_ENUM.ContinueWindow ? {
    title: i18n.ALERT_CLOSE_INFO_MODAL_CONTINUE_SUPPRESSION_WINDOW_TITLE,
    message: continueSuppressionMessageComponent
  } : {
    title: i18n.ALERT_CLOSE_INFO_MODAL_RESTART_SUPPRESSION_TITLE,
    message: restartSuppressionMessageComponent
  };
  return /*#__PURE__*/_react.default.createElement(_eui.EuiConfirmModal, {
    "aria-labelledby": modalTitleId,
    title: title,
    titleProps: {
      id: modalTitleId
    },
    onCancel: onCancel,
    onConfirm: onConfirm,
    confirmButtonText: i18n.ALERT_CLOSE_INFO_MODAL_CONFIRM_LABEL,
    cancelButtonText: i18n.ALERT_CLOSE_INFO_MODAL_CANCEL_LABEL,
    defaultFocusedButton: "confirm",
    "data-test-subj": "alertCloseInfoModal",
    __self: void 0,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 110,
      columnNumber: 5
    }
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
    __self: void 0,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 121,
      columnNumber: 7
    }
  }, message), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "m",
    __self: void 0,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 122,
      columnNumber: 7
    }
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiCheckbox, {
    "data-test-subj": "doNotShowAgainCheckbox",
    id: doNotShowAgainCheckboxId,
    label: i18n.ALERT_CLOSE_INFO_MODAL_DO_NOT_SHOW_AGAIN_LABEL,
    checked: doNotShowAgain,
    onChange: onDoNotShowAgainTicked,
    __self: void 0,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 123,
      columnNumber: 7
    }
  }));
};

/**
 * This hook returns an async function `promptAlertCloseConfirmation` that determines whether to display a confirmation modal when closing an alert.
 *
 * It queries the signals search endpoint to check if the alerts were generated while a suppression window was active in the rule. If so, it shows a modal
 * explaining whether new signals will be suppressed, based on the advanced setting `securitySolution:suppressionBehaviorOnAlertClosure`.
 *
 * If an error occurs during this check, `promptAlertCloseConfirmation` will resolve its promise without blocking alert closure — ensuring the closure process
 * continues even in failure scenarios.
 */
const useAlertCloseInfoModal = () => {
  const [shouldShowModal, setShouldShowModal] = (0, _react.useState)(false);
  const [isModalOpen, setIsModalOpen] = (0, _react.useState)(false);
  const [resolveUserConfirmation, setUserConfirmationResolver] = (0, _react.useState)(() => () => false);
  const {
    overlays,
    services
  } = (0, _kibana.useKibana)();
  const {
    storage
  } = services;
  const promptAlertCloseConfirmation = (0, _react.useCallback)(async params => {
    try {
      if (storage.get(DO_NOT_SHOW_AGAIN_SETTING_KEY)) {
        return Promise.resolve(true);
      }
      if (!(await hasAlertsInSuppressionWindow(params))) {
        return Promise.resolve(true);
      }
      return new Promise(resolvePromise => {
        setUserConfirmationResolver(() => resolvePromise);
        setShouldShowModal(true);
      });
    } catch (error) {
      // We do not want to break alert closure if the logic that decides whether to show the modal breaks
      // users should still be able to close alerts.
      return Promise.resolve(true);
    }
  }, [storage]);
  const handleConfirmationResult = (0, _react.useCallback)(isConfirmed => {
    resolveUserConfirmation(isConfirmed);
    setShouldShowModal(false);
  }, [resolveUserConfirmation]);
  (0, _react.useEffect)(() => {
    if (shouldShowModal && !isModalOpen) {
      setIsModalOpen(true);
      const modalRef = overlays.openModal(/*#__PURE__*/_react.default.createElement(_kibana.KibanaContextProvider, {
        services: services,
        __self: void 0,
        __source: {
          fileName: _jsxFileName,
          lineNumber: 188,
          columnNumber: 9
        }
      }, /*#__PURE__*/_react.default.createElement(AlertCloseConfirmationModal, {
        onConfirmationResult: result => {
          modalRef.close();
          setIsModalOpen(false);
          handleConfirmationResult(result);
        },
        __self: void 0,
        __source: {
          fileName: _jsxFileName,
          lineNumber: 189,
          columnNumber: 11
        }
      })));
    }
  }, [shouldShowModal, overlays, services, isModalOpen, handleConfirmationResult]);
  return {
    promptAlertCloseConfirmation
  };
};
exports.useAlertCloseInfoModal = useAlertCloseInfoModal;