"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerESQLExtensionsRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _projectsSolutionsGroups = require("@kbn/projects-solutions-groups");
var _esqlTypes = require("@kbn/esql-types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Type guard to check if a string is a valid SolutionId.
 * @param str The string to check.
 * @returns True if the string is a valid SolutionId, false otherwise.
 */
function isSolutionId(str) {
  return _projectsSolutionsGroups.KIBANA_PROJECTS.includes(str);
}

/**
 * Registers the ESQL extensions route.
 * This route handles requests for ESQL extensions based on the provided solutionId and query.
 *
 * @param router The IRouter instance to register the route with.
 * @param extensionsRegistry The ESQLExtensionsRegistry instance to use for fetching recommended queries.
 * @param logger The logger instance from the PluginInitializerContext.
 */
const registerESQLExtensionsRoute = (router, extensionsRegistry, {
  logger
}) => {
  router.get({
    path: `${_esqlTypes.REGISTRY_EXTENSIONS_ROUTE}{solutionId}/{query}`,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        solutionId: _configSchema.schema.oneOf([_configSchema.schema.literal('es'), _configSchema.schema.literal('oblt'), _configSchema.schema.literal('security'), _configSchema.schema.literal('chat')], {
          defaultValue: 'oblt' // Default to 'oblt' if no solutionId is provided
        }),
        query: _configSchema.schema.string()
      })
    }
  }, async (requestHandlerContext, request, response) => {
    const core = await requestHandlerContext.core;
    const client = core.elasticsearch.client.asCurrentUser;
    const {
      query,
      solutionId
    } = request.params;
    try {
      var _localSources$indices, _ccsSources$indices, _localSources$aliases, _ccsSources$aliases, _localSources$data_st, _ccsSources$data_stre;
      const localSources = await client.indices.resolveIndex({
        name: '*',
        expand_wildcards: 'open'
      });
      const ccsSources = await client.indices.resolveIndex({
        name: '*:*',
        expand_wildcards: 'open'
      });
      const sources = {
        indices: [...((_localSources$indices = localSources.indices) !== null && _localSources$indices !== void 0 ? _localSources$indices : []), ...((_ccsSources$indices = ccsSources.indices) !== null && _ccsSources$indices !== void 0 ? _ccsSources$indices : [])],
        aliases: [...((_localSources$aliases = localSources.aliases) !== null && _localSources$aliases !== void 0 ? _localSources$aliases : []), ...((_ccsSources$aliases = ccsSources.aliases) !== null && _ccsSources$aliases !== void 0 ? _ccsSources$aliases : [])],
        data_streams: [...((_localSources$data_st = localSources.data_streams) !== null && _localSources$data_st !== void 0 ? _localSources$data_st : []), ...((_ccsSources$data_stre = ccsSources.data_streams) !== null && _ccsSources$data_stre !== void 0 ? _ccsSources$data_stre : [])]
      };

      // Validate solutionId
      const validSolutionId = isSolutionId(solutionId) ? solutionId : 'oblt'; // No solutionId provided, or invalid

      const recommendedQueries = extensionsRegistry.getRecommendedQueries(query, sources, validSolutionId);
      const recommendedFields = extensionsRegistry.getRecommendedFields(query, sources, validSolutionId);
      return response.ok({
        body: {
          recommendedQueries,
          recommendedFields
        }
      });
    } catch (error) {
      logger.get().debug(error);
      throw error;
    }
  });
};
exports.registerESQLExtensionsRoute = registerESQLExtensionsRoute;