"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEndpointSuggestionsRequestHandler = void 0;
exports.registerEndpointSuggestionsRoutes = registerEndpointSuggestionsRoutes;
var _rxjs = require("rxjs");
var _server = require("@kbn/data-plugin/server");
var _terms_enum = require("@kbn/unified-search-plugin/server/autocomplete/terms_enum");
var _terms_agg = require("@kbn/unified-search-plugin/server/autocomplete/terms_agg");
var _common = require("@kbn/fleet-plugin/common");
var _endpoint = require("../../../../common/api/endpoint");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
var _error_handler = require("../error_handler");
var _index_name_utilities = require("../../../../common/endpoint/utils/index_name_utilities");
var _endpoint_metadata_filter = require("../../../../common/endpoint/utils/endpoint_metadata_filter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerEndpointSuggestionsRoutes(router, config$, endpointContext) {
  router.versioned.post({
    access: 'internal',
    path: _constants.SUGGESTIONS_INTERNAL_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '1',
    validate: {
      request: _endpoint.EndpointSuggestionsSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    any: ['canWriteEventFilters', 'canWriteTrustedApplications', 'canWriteTrustedDevices']
  }, endpointContext.logFactory.get('endpointSuggestions'), getEndpointSuggestionsRequestHandler(config$, endpointContext)));
}
const getEndpointSuggestionsRequestHandler = (config$, endpointContext) => {
  return async (context, request, response) => {
    const logger = endpointContext.logFactory.get('suggestions');
    const isTrustedAppsAdvancedModeFFEnabled = endpointContext.experimentalFeatures.trustedAppsAdvancedMode;
    const {
      field: fieldName,
      query,
      filters,
      fieldMeta
    } = request.body;
    let index = '';
    try {
      const config = await (0, _rxjs.firstValueFrom)(config$);
      const {
        savedObjects,
        elasticsearch
      } = await context.core;
      const securitySolutionContext = await context.securitySolution;
      const spaceId = securitySolutionContext.getSpaceId();
      const isSpaceAwarenessEnabled = endpointContext.experimentalFeatures.endpointManagementSpaceAwarenessEnabled;
      let fullFilters = filters ? [...filters] : [];
      let suggestionMethod = _terms_enum.termsEnumSuggestions;
      const suggestionType = request.params.suggestion_type;
      if (suggestionType === 'eventFilters' || isTrustedAppsAdvancedModeFFEnabled && suggestionType === 'trustedApps' || suggestionType === 'trustedDevices') {
        const baseIndexPattern = suggestionType === 'trustedDevices' ? _constants.DEVICE_EVENTS_INDEX_PATTERN : _constants.eventsIndexPattern;
        if (!isSpaceAwarenessEnabled) {
          index = baseIndexPattern;
        } else {
          logger.debug('Using space-aware index pattern');
          const integrationNamespaces = await endpointContext.service.getInternalFleetServices(spaceId).getIntegrationNamespaces(['endpoint']);
          const namespaces = integrationNamespaces.endpoint;
          if (!namespaces || !namespaces.length) {
            logger.error('Failed to retrieve current space index patterns');
            return response.badRequest({
              body: 'Failed to retrieve current space index patterns'
            });
          }
          const indexPattern = namespaces.map(namespace => (0, _index_name_utilities.buildIndexNameWithNamespace)(baseIndexPattern, namespace, {
            preserveWildcard: true
          })).join(',');
          if (indexPattern) {
            logger.debug(`Index pattern to be used: ${indexPattern}`);
            index = indexPattern;
          } else {
            logger.error('Failed to retrieve current space index patterns');
            return response.badRequest({
              body: 'Failed to retrieve current space index patterns'
            });
          }
        }
      } else if (suggestionType === 'endpoints') {
        suggestionMethod = _terms_agg.termsAggSuggestions;
        index = _constants.METADATA_UNITED_INDEX;
        const agentPolicyIds = [];
        const fleetService = securitySolutionContext.getInternalFleetServices();
        const endpointPackagePolicies = await fleetService.packagePolicy.fetchAllItems(savedObjects.client, {
          kuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:endpoint`,
          spaceIds: isSpaceAwarenessEnabled ? [spaceId] : ['*']
        });
        for await (const batch of endpointPackagePolicies) {
          const batchAgentPolicyIds = batch.flatMap(policy => policy.policy_ids);
          agentPolicyIds.push(...batchAgentPolicyIds);
        }
        const baseFilters = (0, _endpoint_metadata_filter.buildBaseEndpointMetadataFilter)(agentPolicyIds);
        fullFilters = [...fullFilters, baseFilters];
      } else {
        return response.badRequest({
          body: `Invalid suggestion_type: ${suggestionType}`
        });
      }
      const abortSignal = (0, _server.getRequestAbortedSignal)(request.events.aborted$);
      const body = await suggestionMethod(config, savedObjects.client, elasticsearch.client.asInternalUser, index, fieldName, query, fullFilters, fieldMeta, abortSignal);
      return response.ok({
        body
      });
    } catch (error) {
      return (0, _error_handler.errorHandler)(logger, response, error);
    }
  };
};
exports.getEndpointSuggestionsRequestHandler = getEndpointSuggestionsRequestHandler;