"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setEndpointListApiMockImplementation = exports.mockEndpointResultList = exports.mockEndpointDetailsApiResult = void 0;
var _types = require("../../../../../common/endpoint/types");
var _generate_data = require("../../../../../common/endpoint/generate_data");
var _ingest = require("../../../services/policies/ingest");
var _mocks = require("../../../../common/lib/endpoint/endpoint_pending_actions/mocks");
var _constants = require("../../../../../common/endpoint/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const generator = new _generate_data.EndpointDocGenerator('seed');
const mockEndpointResultList = (options = {}) => {
  const {
    total = 1,
    page = 0,
    pageSize = 10
  } = options;

  // Skip any that are before the page we're on
  const numberToSkip = pageSize * page;

  // total - numberToSkip is the count of non-skipped ones, but return no more than a pageSize, and no less than 0
  const actualCountToReturn = Math.max(Math.min(total - numberToSkip, pageSize), 0);
  const hosts = [];
  for (let index = 0; index < actualCountToReturn; index++) {
    const newDate = new Date();
    const metadata = generator.generateHostMetadata(newDate.getTime());
    hosts.push({
      metadata,
      host_status: _types.HostStatus.UNHEALTHY,
      last_checkin: newDate.toISOString()
    });
  }
  const mock = {
    data: hosts,
    total,
    page,
    pageSize,
    sortDirection: _constants.ENDPOINT_DEFAULT_SORT_DIRECTION,
    sortField: _constants.ENDPOINT_DEFAULT_SORT_FIELD
  };
  return mock;
};

/**
 * returns a mocked API response for retrieving a single host metadata
 */
exports.mockEndpointResultList = mockEndpointResultList;
const mockEndpointDetailsApiResult = () => {
  const newDate = new Date();
  const metadata = generator.generateHostMetadata(newDate.getTime());
  return {
    metadata,
    host_status: _types.HostStatus.UNHEALTHY,
    last_checkin: newDate.toISOString()
  };
};

/**
 * Mock API handlers used by the Endpoint Host list. It also sets up a list of
 * API handlers for Host details based on a list of Host results.
 */
exports.mockEndpointDetailsApiResult = mockEndpointDetailsApiResult;
const endpointListApiPathHandlerMocks = ({
  endpointsResults = mockEndpointResultList({
    total: 3
  }).data,
  epmPackages = [generator.generateEpmPackage()],
  endpointPackagePolicies = [],
  policyResponse = generator.generatePolicyResponse(),
  agentPolicy = generator.generateAgentPolicy(),
  totalAgentsUsingEndpoint = 0,
  transforms = []
} = {}) => {
  const apiHandlers = {
    // endpoint package info
    [_ingest.INGEST_API_EPM_PACKAGES]: () => {
      return {
        items: epmPackages
      };
    },
    // endpoint list
    [_constants.HOST_METADATA_LIST_ROUTE]: () => {
      return {
        data: endpointsResults,
        total: (endpointsResults === null || endpointsResults === void 0 ? void 0 : endpointsResults.length) || 0,
        page: 0,
        pageSize: 10,
        sortDirection: _constants.ENDPOINT_DEFAULT_SORT_DIRECTION,
        sortField: _constants.ENDPOINT_DEFAULT_SORT_FIELD
      };
    },
    // Do policies reference in endpoint list exist
    // just returns 1 single agent policy that includes all the packagePolicy IDs provided
    [_ingest.INGEST_API_AGENT_POLICIES]: () => {
      return {
        items: [agentPolicy],
        total: 10,
        perPage: 10,
        page: 1
      };
    },
    // Policy Response
    '/api/endpoint/policy_response': () => {
      return {
        policy_response: policyResponse
      };
    },
    // List of Policies (package policies) for onboarding
    [_ingest.INGEST_API_PACKAGE_POLICIES]: () => {
      return {
        items: endpointPackagePolicies,
        page: 1,
        perPage: 10,
        total: endpointPackagePolicies === null || endpointPackagePolicies === void 0 ? void 0 : endpointPackagePolicies.length
      };
    },
    // List of Policies (package policies) for onboarding
    [`${_ingest.INGEST_API_PACKAGE_POLICIES}/_bulk_get`]: () => {
      return {
        items: endpointPackagePolicies
      };
    },
    // List of Agents using Endpoint
    [_ingest.INGEST_API_FLEET_AGENTS]: () => {
      return {
        total: totalAgentsUsingEndpoint,
        items: [],
        page: 1,
        perPage: 10
      };
    },
    // Pending Actions
    [_constants.ACTION_STATUS_ROUTE]: () => {
      return (0, _mocks.pendingActionsResponseMock)();
    },
    [_constants.METADATA_TRANSFORMS_STATUS_INTERNAL_ROUTE]: () => ({
      count: transforms.length,
      transforms
    })
  };

  // Build a GET route handler for each endpoint details based on the list of Endpoints passed on input
  if (endpointsResults) {
    endpointsResults.forEach(host => {
      // @ts-expect-error
      apiHandlers[`${_constants.HOST_METADATA_LIST_ROUTE}/${host.metadata.agent.id}`] = () => host;
    });
  }
  return apiHandlers;
};

/**
 * Sets up mock implementations in support of the Endpoints list view
 *
 * @param mockedHttpService
 * @param endpointsResults
 * @param pathHandlersOptions
 */
const setEndpointListApiMockImplementation = (mockedHttpService, {
  endpointsResults = mockEndpointResultList({
    total: 3
  }).data,
  ...pathHandlersOptions
} = {}) => {
  const apiHandlers = endpointListApiPathHandlerMocks({
    ...pathHandlersOptions,
    endpointsResults
  });

  // Setup handling of GET requests
  mockedHttpService.get.mockImplementation(async (...args) => {
    const [path] = args;
    if (typeof path === 'string') {
      if (apiHandlers[path]) {
        // @ts-expect-error upgrade typescript v5.4.5
        return apiHandlers[path]();
      }
    }
    throw new Error(`MOCK: api request does not have a mocked handler: ${path}`);
  });
  mockedHttpService.post.mockImplementation(async (...args) => {
    const [path] = args;
    if (typeof path === 'string') {
      if (apiHandlers[path]) {
        // @ts-expect-error upgrade typescript v5.4.5
        return apiHandlers[path]();
      }
    }
    throw new Error(`MOCK: api request does not have a mocked handler: ${path}`);
  });
};
exports.setEndpointListApiMockImplementation = setEndpointListApiMockImplementation;