"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RunScheduledReportTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _spacesUtils = require("@kbn/spaces-utils");
var _reportingServer = require("@kbn/reporting-server");
var _reportingCommon = require("@kbn/reporting-common");
var _ = require(".");
var _error_logger = require("./error_logger");
var _saved_objects = require("../../saved_objects");
var _run_report = require("./run_report");
var _scheduled_report = require("../store/scheduled_report");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_ATTACHMENT_SIZE = 10 * 1024 * 1024; // 10mb

class RunScheduledReportTask extends _run_report.RunReportTask {
  constructor(...args) {
    super(...args);
    (0, _defineProperty2.default)(this, "exportType", _reportingCommon.EXPORT_TYPE_SCHEDULED);
  }
  get TYPE() {
    return _.SCHEDULED_REPORTING_EXECUTE_TYPE;
  }
  async prepareJob(taskInstance) {
    var _report;
    const {
      runAt,
      params: scheduledReportTaskParams
    } = taskInstance;
    let report;
    let jobId;
    let scheduledReport;
    const task = scheduledReportTaskParams;
    const scheduledReportId = task.id;
    const reportSpaceId = task.spaceId || _spacesUtils.DEFAULT_SPACE_ID;
    try {
      if (!scheduledReportId) {
        throw new Error(`Invalid scheduled_report saved object data provided in scheduled task! - No saved object with id "${scheduledReportId}"`);
      }
      const internalSoClient = await this.opts.reporting.getInternalSoClient();
      scheduledReport = await internalSoClient.get(_saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE, scheduledReportId, {
        namespace: reportSpaceId
      });
      const store = await this.opts.reporting.getStore();

      // Add the report to ReportingStore to show as processing
      report = await store.addReport(new _scheduled_report.ScheduledReport({
        runAt,
        kibanaId: this.kibanaId,
        kibanaName: this.kibanaName,
        queueTimeout: this.getQueueTimeout().asMilliseconds(),
        scheduledReport,
        spaceId: reportSpaceId
      }));
      jobId = report._id;
      if (!jobId) {
        throw new Error(`Unable to store report document in ReportingStore`);
      }

      // event tracking of claimed job
      const eventTracker = this.getEventTracker(report);
      const timeSinceCreation = Date.now() - new Date(report.created_at).valueOf();
      eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.claimJob({
        timeSinceCreation,
        scheduledTaskId: report.scheduled_report_id,
        scheduleType: _reportingServer.ScheduleType.SCHEDULED
      });
    } catch (failedToClaim) {
      // error claiming report - log the error
      (0, _error_logger.errorLogger)(this.logger, `Error in running scheduled report ${scheduledReportId}`, failedToClaim);
    }
    return {
      jobId: jobId,
      report,
      task: (_report = report) === null || _report === void 0 ? void 0 : _report.toReportTaskJSON(),
      scheduledReport
    };
  }
  getMaxAttempts() {
    var _this$opts$config$cap;
    const maxAttempts = (_this$opts$config$cap = this.opts.config.capture.maxAttempts) !== null && _this$opts$config$cap !== void 0 ? _this$opts$config$cap : 1;
    return {
      maxTaskAttempts: 1,
      maxRetries: maxAttempts - 1
    };
  }
  async notify(report, taskInstance, output, byteSize, scheduledReport, spaceId) {
    try {
      const {
        runAt,
        params
      } = taskInstance;
      const task = params;
      if (!scheduledReport) {
        const internalSoClient = await this.opts.reporting.getInternalSoClient();
        scheduledReport = await internalSoClient.get(_saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE, task.id, {
          namespace: spaceId
        });
      }
      const {
        notification
      } = scheduledReport.attributes;
      if (notification && notification.email) {
        if (byteSize > MAX_ATTACHMENT_SIZE) {
          throw new Error('The report is larger than the 10MB limit.');
        }
        if (!this.emailNotificationService) {
          throw new Error('Reporting notification service has not been initialized.');
        }
        const email = notification.email;
        const title = scheduledReport.attributes.title;
        const extension = this.getJobContentExtension(report.jobtype);
        await this.emailNotificationService.notify({
          reporting: this.opts.reporting,
          index: report._index,
          id: report._id,
          filename: `${title}-${runAt.toISOString()}.${extension}`,
          contentType: output.content_type,
          relatedObject: {
            id: scheduledReport.id,
            type: scheduledReport.type,
            namespace: spaceId
          },
          emailParams: {
            to: email.to,
            cc: email.cc,
            bcc: email.bcc,
            subject: `${title}-${runAt.toISOString()} scheduled report`,
            spaceId
          }
        });

        // event tracking of successful notification
        const eventTracker = this.getEventTracker(report);
        eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.completeNotification({
          byteSize,
          scheduledTaskId: report.scheduled_report_id,
          scheduleType: _reportingServer.ScheduleType.SCHEDULED
        });
      }
    } catch (error) {
      const message = `Error sending notification for scheduled report: ${error.message}`;
      // event tracking of successful notification
      const eventTracker = this.getEventTracker(report);
      eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.failedNotification({
        byteSize,
        scheduledTaskId: report.scheduled_report_id,
        scheduleType: _reportingServer.ScheduleType.SCHEDULED,
        errorMessage: message
      });
      this.saveExecutionWarning(report, {
        ...output,
        size: byteSize
      }, message).catch(failedToSaveWarning => {
        (0, _error_logger.errorLogger)(this.logger, `Error in saving execution warning ${report._id}`, failedToSaveWarning);
      });
    }
  }
  getTaskDefinition() {
    const queueTimeout = this.getQueueTimeoutAsInterval();
    const maxConcurrency = this.getMaxConcurrency();
    return {
      type: _.SCHEDULED_REPORTING_EXECUTE_TYPE,
      title: 'Reporting: execute scheduled job',
      createTaskRunner: this.getTaskRunner(),
      timeout: queueTimeout,
      maxConcurrency
    };
  }
  async scheduleTask(request, params) {
    const spaceId = this.opts.reporting.getSpaceId(request, this.logger);
    const taskInstance = {
      id: params.id,
      taskType: _.SCHEDULED_REPORTING_EXECUTE_TYPE,
      state: {},
      params: {
        id: params.id,
        spaceId: spaceId || _spacesUtils.DEFAULT_SPACE_ID,
        jobtype: params.jobtype
      },
      schedule: params.schedule
    };
    return await this.getTaskManagerStart().schedule(taskInstance, {
      request
    });
  }
}
exports.RunScheduledReportTask = RunScheduledReportTask;