"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.timeSeriesSearchServiceFactory = timeSeriesSearchServiceFactory;
exports.useTimeSeriesSearchService = useTimeSeriesSearchService;
var _react = require("react");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _results_service = require("../../services/results_service");
var _chart_config_builder = require("../../util/chart_config_builder");
var _job_utils = require("../../../../common/util/job_utils");
var _kibana = require("../../contexts/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function timeSeriesSearchServiceFactory(mlResultsService, mlApi) {
  function getMetricData(job, detectorIndex, entityFields, earliestMs, latestMs, intervalMs, esMetricFunction) {
    if ((0, _job_utils.isModelPlotChartableForDetector)(job, detectorIndex) && (0, _job_utils.isModelPlotEnabled)(job, detectorIndex, entityFields)) {
      // Extract the partition, by, over fields on which to filter.
      const criteriaFields = [];
      const detector = job.analysis_config.detectors[detectorIndex];
      if (detector.partition_field_name !== undefined) {
        const partitionEntity = (0, _lodash.find)(entityFields, {
          fieldName: detector.partition_field_name
        });
        if (partitionEntity !== undefined) {
          criteriaFields.push({
            fieldName: 'partition_field_name',
            fieldValue: partitionEntity.fieldName
          }, {
            fieldName: 'partition_field_value',
            fieldValue: partitionEntity.fieldValue
          });
        }
      }
      if (detector.over_field_name !== undefined) {
        const overEntity = (0, _lodash.find)(entityFields, {
          fieldName: detector.over_field_name
        });
        if (overEntity !== undefined) {
          criteriaFields.push({
            fieldName: 'over_field_name',
            fieldValue: overEntity.fieldName
          }, {
            fieldName: 'over_field_value',
            fieldValue: overEntity.fieldValue
          });
        }
      }
      if (detector.by_field_name !== undefined) {
        const byEntity = (0, _lodash.find)(entityFields, {
          fieldName: detector.by_field_name
        });
        if (byEntity !== undefined) {
          criteriaFields.push({
            fieldName: 'by_field_name',
            fieldValue: byEntity.fieldName
          }, {
            fieldName: 'by_field_value',
            fieldValue: byEntity.fieldValue
          });
        }
      }
      return mlResultsService.getModelPlotOutput(job.job_id, detectorIndex, criteriaFields, earliestMs, latestMs, intervalMs);
    } else {
      const obj = {
        success: true,
        results: {}
      };
      const chartConfig = (0, _chart_config_builder.buildConfigFromDetector)(job, detectorIndex);
      return mlResultsService.getMetricData(chartConfig.datafeedConfig.indices.join(','), entityFields, chartConfig.datafeedConfig.query, esMetricFunction !== null && esMetricFunction !== void 0 ? esMetricFunction : chartConfig.metricFunction, chartConfig.metricFieldName, chartConfig.summaryCountFieldName, chartConfig.timeField, earliestMs, latestMs, intervalMs, chartConfig === null || chartConfig === void 0 ? void 0 : chartConfig.datafeedConfig).pipe((0, _rxjs.map)(resp => {
        (0, _lodash.each)(resp.results, (value, time) => {
          // @ts-ignore
          obj.results[time] = {
            actual: value
          };
        });
        return obj;
      }));
    }
  }
  /**
   * Builds chart detail information (charting function description and entity counts) used
   * in the title area of the time series chart.
   * Queries Elasticsearch if necessary to obtain the distinct count of entities
   * for which data is being plotted.
   * @param job Job config info
   * @param detectorIndex The index of the detector in the job config
   * @param entityFields Array of field name - field value pairs
   * @param earliestMs Earliest timestamp in milliseconds
   * @param latestMs Latest timestamp in milliseconds
   * @param metricFunctionDescription The underlying function (min, max, avg) for "metric" detector type
   * @returns chart data to plot for Single Metric Viewer/Time series explorer
   */
  function getChartDetails(job, detectorIndex, entityFields, earliestMs, latestMs, metricFunctionDescription) {
    return new Promise((resolve, reject) => {
      const obj = {
        success: true,
        results: {
          functionLabel: '',
          entityData: {
            entities: []
          }
        }
      };
      const chartConfig = (0, _chart_config_builder.buildConfigFromDetector)(job, detectorIndex, metricFunctionDescription);
      let functionLabel = chartConfig.metricFunction;
      if (chartConfig.metricFieldName !== undefined) {
        functionLabel += ` ${chartConfig.metricFieldName}`;
      }
      obj.results.functionLabel = functionLabel;
      const blankEntityFields = (0, _lodash.filter)(entityFields, entity => {
        return entity.fieldValue === null;
      });

      // Look to see if any of the entity fields have defined values
      // (i.e. blank input), and if so obtain the cardinality.
      if (blankEntityFields.length === 0) {
        obj.results.entityData.count = 1;
        obj.results.entityData.entities = entityFields;
        resolve(obj);
      } else {
        const entityFieldNames = blankEntityFields.map(f => f.fieldName);
        mlApi.getCardinalityOfFields({
          index: chartConfig.datafeedConfig.indices.join(','),
          fieldNames: entityFieldNames,
          query: chartConfig.datafeedConfig.query,
          timeFieldName: chartConfig.timeField,
          earliestMs,
          latestMs
        }).then(results => {
          (0, _lodash.each)(blankEntityFields, field => {
            // results will not contain keys for non-aggregatable fields,
            // so store as 0 to indicate over all field values.
            obj.results.entityData.entities.push({
              fieldName: field.fieldName,
              cardinality: (0, _lodash.get)(results, field.fieldName, 0)
            });
          });
          resolve(obj);
        }).catch(resp => {
          reject(resp);
        });
      }
    });
  }
  return {
    getMetricData,
    getChartDetails
  };
}
function useTimeSeriesSearchService() {
  const {
    services: {
      mlServices: {
        mlApi
      }
    }
  } = (0, _kibana.useMlKibana)();
  const mlResultsService = (0, _results_service.mlResultsServiceProvider)(mlApi);
  const mlForecastService = (0, _react.useMemo)(() => timeSeriesSearchServiceFactory(mlResultsService, mlApi), [mlApi, mlResultsService]);
  return mlForecastService;
}