"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAggDescriptor = createAggDescriptor;
exports.createTileMapLayerDescriptor = createTileMapLayerDescriptor;
var _constants = require("../../../common/constants");
var _vector_style = require("../styles/vector/vector_style");
var _es_geo_grid_source = require("../sources/es_geo_grid_source");
var _vector_layer = require("./vector_layer");
var _heatmap_layer = require("./heatmap_layer");
var _vector_style_defaults = require("../styles/vector/vector_style_defaults");
var _color_palettes = require("../styles/color_palettes");
var _get_agg_key = require("../../../common/get_agg_key");
var _is_metric_countable = require("../util/is_metric_countable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultDynamicProperties = (0, _vector_style_defaults.getDefaultDynamicProperties)();
function isHeatmap(mapType) {
  return mapType.toLowerCase() === 'heatmap';
}
function getGeoGridRequestType(mapType) {
  if (isHeatmap(mapType)) {
    return _constants.RENDER_AS.HEATMAP;
  }
  if (mapType.toLowerCase() === 'shaded geohash grid') {
    return _constants.RENDER_AS.GRID;
  }
  return _constants.RENDER_AS.POINT;
}
function createAggDescriptor(mapType, metricAgg, metricFieldName) {
  const aggTypeKey = Object.keys(_constants.AGG_TYPE).find(key => {
    return _constants.AGG_TYPE[key] === metricAgg;
  });
  const aggType = aggTypeKey ? _constants.AGG_TYPE[aggTypeKey] : undefined;
  if (!aggType || aggType === _constants.AGG_TYPE.COUNT || !metricFieldName || isHeatmap(mapType) && !(0, _is_metric_countable.isMetricCountable)(aggType)) {
    return {
      type: _constants.AGG_TYPE.COUNT
    };
  }
  return aggType === _constants.AGG_TYPE.PERCENTILE ? {
    type: aggType,
    field: metricFieldName,
    percentile: _constants.DEFAULT_PERCENTILE
  } : {
    type: aggType,
    field: metricFieldName
  };
}
function createTileMapLayerDescriptor({
  label,
  mapType,
  colorSchema,
  indexPatternId,
  geoFieldName,
  metricAgg,
  metricFieldName
}) {
  if (!indexPatternId || !geoFieldName) {
    return null;
  }
  const metricsDescriptor = createAggDescriptor(mapType, metricAgg, metricFieldName);
  const geoGridSourceDescriptor = _es_geo_grid_source.ESGeoGridSource.createDescriptor({
    indexPatternId,
    geoField: geoFieldName,
    metrics: [metricsDescriptor],
    requestType: getGeoGridRequestType(mapType),
    resolution: _constants.GRID_RESOLUTION.MOST_FINE
  });
  if (isHeatmap(mapType)) {
    return _heatmap_layer.HeatmapLayer.createDescriptor({
      label,
      sourceDescriptor: geoGridSourceDescriptor
    });
  }
  const metricSourceKey = (0, _get_agg_key.getSourceAggKey)({
    aggType: metricsDescriptor.type,
    aggFieldName: 'field' in metricsDescriptor ? metricsDescriptor.field : ''
  });
  const metricStyleField = {
    name: metricSourceKey,
    origin: _constants.FIELD_ORIGIN.SOURCE
  };
  const colorPallette = _color_palettes.NUMERICAL_COLOR_PALETTES.find(pallette => {
    return pallette.value.toLowerCase() === colorSchema.toLowerCase();
  });
  const styleProperties = {
    [_constants.VECTOR_STYLES.FILL_COLOR]: {
      type: _constants.STYLE_TYPE.DYNAMIC,
      options: {
        ...defaultDynamicProperties[_constants.VECTOR_STYLES.FILL_COLOR].options,
        field: metricStyleField,
        color: colorPallette ? colorPallette.value : 'Yellow to Red',
        type: _constants.COLOR_MAP_TYPE.ORDINAL,
        fieldMetaOptions: {
          ...defaultDynamicProperties[_constants.VECTOR_STYLES.FILL_COLOR].options.fieldMetaOptions,
          isEnabled: false
        }
      }
    },
    [_constants.VECTOR_STYLES.LINE_COLOR]: {
      type: _constants.STYLE_TYPE.STATIC,
      options: {
        color: '#3d3d3d'
      }
    }
  };
  if (mapType.toLowerCase() === 'scaled circle markers') {
    styleProperties[_constants.VECTOR_STYLES.ICON_SIZE] = {
      type: _constants.STYLE_TYPE.DYNAMIC,
      options: {
        ...defaultDynamicProperties[_constants.VECTOR_STYLES.ICON_SIZE].options,
        maxSize: 18,
        field: metricStyleField,
        fieldMetaOptions: {
          ...defaultDynamicProperties[_constants.VECTOR_STYLES.ICON_SIZE].options.fieldMetaOptions,
          isEnabled: false
        }
      }
    };
  }
  return _vector_layer.GeoJsonVectorLayer.createDescriptor({
    label,
    sourceDescriptor: geoGridSourceDescriptor,
    style: _vector_style.VectorStyle.createDescriptor(styleProperties)
  });
}