"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getGrokProcessor = getGrokProcessor;
var _constants = require("../constants");
var _get_review_fields = require("./get_review_fields");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Generates a GROK processor definition by combining extracted tokens and
 * the result of an LLM review. It constructs a root GROK pattern and
 * optionally defines custom pattern definitions for fields with multiple
 * columns, ensuring patterns are validated and adjusted based on example
 * values.
 */
function getGrokProcessor(nodes, reviewResult) {
  let rootPattern = '';
  const patternDefinitions = {};
  let targetDefinition;
  const appendNode = node => {
    if (targetDefinition) {
      if (!patternDefinitions[targetDefinition]) {
        patternDefinitions[targetDefinition] = '';
      }
      patternDefinitions[targetDefinition] += (0, _utils.isNamedField)(node) ? `%{${node.component}}` : (0, _get_review_fields.sanitize)(node.pattern);
    } else {
      rootPattern += (0, _utils.isNamedField)(node) ? `%{${node.component}:${node.id}}` : (0, _get_review_fields.sanitize)(node.pattern);
    }
  };
  const pickValidPattern = (suggestedPattern, originalPattern, exampleValues) => {
    // If the suggested pattern is a collapsible token, return the original pattern. These have been vetted by the heuristics to not be unecessarily greedy, something the LLM does not do well, so should not be changed
    if ((0, _get_review_fields.isCollapsiblePattern)(suggestedPattern)) {
      return originalPattern;
    }
    // If the suggested pattern does not match any of the example values, return the original pattern
    if (suggestedPattern !== originalPattern && exampleValues.some(val => !_constants.GROK_REGEX_MAP[suggestedPattern] || !_constants.GROK_REGEX_MAP[suggestedPattern].complete.test(val))) {
      return originalPattern;
    }
    return suggestedPattern;
  };
  nodes.forEach(node => {
    if ((0, _utils.isNamedField)(node)) {
      const match = reviewResult.fields.find(field => field.columns.includes(node.id));
      if (match) {
        if (match.columns.length >= 2) {
          // Node is part of a group with multiple columns, create a custom pattern definition and add the token to it
          const index = match.columns.indexOf(node.id);
          const patternDefinitionName = `CUSTOM_${match.name.replace(/^(resource\.)?(attributes\.)?(custom_)?/g, '').toUpperCase().replace(/\W+/g, '_').replace(/^_|_$/g, '')}`;
          if (index === 0) {
            appendNode({
              id: match.name,
              component: patternDefinitionName,
              values: []
            });

            // Change destination to custom pattern destination
            targetDefinition = patternDefinitionName;
          }

          // Append the token to the current pattern definition
          appendNode({
            id: match.name,
            component: pickValidPattern(match.grok_components[index], node.component, node.values),
            values: node.values
          });
          if (index === match.columns.length - 1) {
            // Change destination back after the last column
            targetDefinition = undefined;
          }
        } else {
          // Token is part of a single column field, change the token according to the feedback and add to the root pattern
          appendNode({
            id: match.name,
            component: pickValidPattern(match.grok_components[0], node.component, node.values),
            values: node.values
          });
        }
      } else {
        // Token is a field but did not get reviewed, keep as is and add the token to the current pattern definition
        appendNode(node);
      }
    } else {
      // Token is a separator character, keep as is and add to the current pattern definition
      appendNode(node);
    }
  });
  return {
    description: reviewResult.log_source,
    patterns: [rootPattern],
    pattern_definitions: patternDefinitions
  };
}