"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkMigrateAgents = bulkMigrateAgents;
exports.migrateSingleAgent = migrateSingleAgent;
var _services = require("../../../common/services");
var _errors = require("../../errors");
var _get_current_namespace = require("../spaces/get_current_namespace");
var _constants = require("../../constants");
var _ = require("..");
var _constants2 = require("../../../common/constants");
var _actions = require("./actions");
var _crud = require("./crud");
var _migrate_action_runner = require("./migrate_action_runner");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function migrateSingleAgent(esClient, soClient, agentId, agentPolicy, agent, options) {
  var _agent$components;
  // Check the user has the correct license
  if (!_.licenseService.hasAtLeast(_constants2.LICENSE_FOR_AGENT_MIGRATION)) {
    throw new _errors.FleetUnauthorizedError(`Agent migration requires an ${_constants2.LICENSE_FOR_AGENT_MIGRATION} license. Please upgrade your license.`);
  }

  //  If the agent belongs to a policy that is protected or has fleet-server as a component meaning its a fleet server agent, throw an error
  if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_protected) {
    throw new _errors.FleetUnauthorizedError(`Agent is protected and cannot be migrated`);
  }
  if ((_agent$components = agent.components) !== null && _agent$components !== void 0 && _agent$components.some(c => c.type === 'fleet-server')) {
    throw new _errors.FleetUnauthorizedError(`Fleet server agents cannot be migrated`);
  }
  if (!(0, _services.isAgentMigrationSupported)(agent)) {
    var _agent$local_metadata, _agent$local_metadata2, _agent$local_metadata3;
    // Check if it's specifically a containerized agent
    if (((_agent$local_metadata = agent.local_metadata) === null || _agent$local_metadata === void 0 ? void 0 : (_agent$local_metadata2 = _agent$local_metadata.elastic) === null || _agent$local_metadata2 === void 0 ? void 0 : (_agent$local_metadata3 = _agent$local_metadata2.agent) === null || _agent$local_metadata3 === void 0 ? void 0 : _agent$local_metadata3.upgradeable) === false) {
      throw new _errors.FleetError(`Containerized agents cannot be migrated`);
    }
    // Otherwise it's a version issue
    throw new _errors.FleetError(`Agent cannot be migrated. Migrate action is supported from version ${_services.MINIMUM_MIGRATE_AGENT_VERSION}.`);
  }
  const response = await (0, _actions.createAgentAction)(esClient, soClient, {
    agents: [agentId],
    created_at: new Date().toISOString(),
    type: 'MIGRATE',
    policyId: options.policyId,
    data: {
      target_uri: options.uri,
      settings: options.settings
    },
    ...(options.enrollment_token && {
      secrets: {
        enrollment_token: options.enrollment_token
      }
    })
  });
  return {
    actionId: response.id
  };
}
async function bulkMigrateAgents(esClient, soClient, options) {
  var _options$batchSize;
  // Check the user has the correct license
  if (!_.licenseService.hasAtLeast(_constants2.LICENSE_FOR_AGENT_MIGRATION)) {
    throw new _errors.FleetUnauthorizedError(`Agent migration requires an ${_constants2.LICENSE_FOR_AGENT_MIGRATION} license. Please upgrade your license.`);
  }
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  if ('agentIds' in options) {
    const givenAgents = await (0, _crud.getAgents)(esClient, soClient, options);
    return await (0, _migrate_action_runner.bulkMigrateAgentsBatch)(esClient, soClient, givenAgents, {
      enrollment_token: options.enrollment_token,
      uri: options.uri,
      settings: options.settings,
      spaceId: currentSpaceId
    });
  }
  const batchSize = (_options$batchSize = options.batchSize) !== null && _options$batchSize !== void 0 ? _options$batchSize : _constants.SO_SEARCH_LIMIT;
  const res = await (0, _crud.getAgentsByKuery)(esClient, soClient, {
    kuery: options.kuery,
    spaceId: currentSpaceId,
    showInactive: false,
    page: 1,
    perPage: batchSize
  });
  if (res.total <= batchSize) {
    return await (0, _migrate_action_runner.bulkMigrateAgentsBatch)(esClient, soClient, res.agents, {
      enrollment_token: options.enrollment_token,
      uri: options.uri,
      settings: options.settings,
      spaceId: currentSpaceId
    });
  } else {
    return await new _migrate_action_runner.MigrateActionRunner(esClient, soClient, {
      ...options,
      batchSize,
      total: res.total,
      spaceId: currentSpaceId
    }, {
      pitId: await (0, _crud.openPointInTime)(esClient)
    }).runActionAsyncTask();
  }
}