"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.remove = exports.findByName = exports.find = exports.append = void 0;
var _builder = require("../../../../builder");
var _visitor = require("../../../../visitor");
var commands = _interopRequireWildcard(require(".."));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns the first command option AST node that satisfies the predicate.
 *
 * @param command The command AST node to search for options.
 * @param predicate The predicate to filter options.
 * @returns The option found in the command, if any.
 */
const find = (command, predicate) => {
  return new _visitor.Visitor().on('visitCommand', ctx => {
    for (const opt of ctx.options()) {
      if (predicate(opt)) {
        return opt;
      }
    }
    return undefined;
  }).visitCommand(command);
};

/**
 * Returns the first command option AST node with a given name in the query.
 *
 * @param ast The root AST node to search for command options.
 * @param commandName Command name to search for.
 * @param optionName Option name to search for.
 * @returns The option found in the command, if any.
 */
exports.find = find;
const findByName = (ast, commandName, optionName) => {
  const command = commands.find(ast, cmd => cmd.name === commandName);
  if (!command) {
    return undefined;
  }
  return find(command, opt => opt.name === optionName);
};

/**
 * Inserts a command option into the command's arguments list. The option can
 * be specified as a string or an AST node.
 *
 * @param command The command AST node to insert the option into.
 * @param option The option to insert.
 * @returns The inserted option.
 */
exports.findByName = findByName;
const append = (command, option) => {
  if (typeof option === 'string') {
    option = _builder.Builder.option({
      name: option
    });
  }
  command.args.push(option);
  return option;
};

/**
 * Removes the first command option from the command's arguments list that
 * satisfies the predicate.
 *
 * @param command The command AST node to remove the option from.
 * @param predicate The predicate to filter options.
 * @returns The removed option, if any.
 */
exports.append = append;
const remove = (ast, option) => {
  return new _visitor.Visitor().on('visitCommandOption', ctx => {
    return ctx.node === option;
  }).on('visitCommand', ctx => {
    let target;
    for (const opt of ctx.options()) {
      if (opt === option) {
        target = opt;
        break;
      }
    }
    if (!target) {
      return false;
    }
    const index = ctx.node.args.indexOf(target);
    if (index === -1) {
      return false;
    }
    ctx.node.args.splice(index, 1);
    return true;
  }).on('visitQuery', ctx => {
    for (const success of ctx.visitCommands()) {
      if (success) {
        return true;
      }
    }
    return false;
  }).visitQuery(ast);
};
exports.remove = remove;