"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerReindexIndicesRoutes = registerReindexIndicesRoutes;
exports.reindexSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _elasticsearch = require("@elastic/elasticsearch");
var _server = require("@kbn/es-ui-shared-plugin/server");
var _upgradeAssistantPkgServer = require("@kbn/upgrade-assistant-pkg-server");
var _common = require("../../../common");
var _map_any_error_to_kibana_http_response = require("./map_any_error_to_kibana_http_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const reindexSchema = exports.reindexSchema = _configSchema.schema.object({
  indexName: _configSchema.schema.string(),
  newIndexName: _configSchema.schema.string(),
  reindexOptions: _configSchema.schema.maybe(_configSchema.schema.object({
    enqueue: _configSchema.schema.maybe(_configSchema.schema.boolean()),
    deleteOldIndex: _configSchema.schema.maybe(_configSchema.schema.boolean())
  })),
  settings: _configSchema.schema.maybe(_configSchema.schema.object({
    mode: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('standard'), _configSchema.schema.literal('lookup'), _configSchema.schema.literal('logsdb'), _configSchema.schema.literal('time_series')]))
  }))
});
function registerReindexIndicesRoutes({
  router,
  getReindexService
}) {
  // Start reindex for an index
  router.post({
    path: _common.REINDEX_SERVICE_BASE_PATH,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es and saved object clients for authorization'
      }
    },
    validate: {
      body: reindexSchema
    }
  }, async ({
    core
  }, request, response) => {
    const {
      savedObjects: {
        getClient
      },
      elasticsearch: {
        client: esClient
      }
    } = await core;
    if (request.body.newIndexName.trim().length === 0) {
      return response.badRequest({
        body: 'New index name cannot be empty'
      });
    }
    try {
      const reindexService = (await getReindexService()).getScopedClient({
        savedObjects: getClient({
          includedHiddenTypes: [_upgradeAssistantPkgServer.REINDEX_OP_TYPE]
        }),
        dataClient: esClient,
        request
      });
      const result = await reindexService.reindexOrResume({
        ...request.body,
        newIndexName: request.body.newIndexName.trim()
      });
      return response.ok({
        body: result
      });
    } catch (error) {
      if (error instanceof _elasticsearch.errors.ResponseError) {
        return (0, _server.handleEsError)({
          error,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(error);
    }
  });

  // Get status
  router.get({
    path: `${_common.REINDEX_SERVICE_BASE_PATH}/{indexName}`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es and saved object clients for authorization'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, async ({
    core
  }, request, response) => {
    const {
      savedObjects,
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      getClient
    } = savedObjects;
    const {
      indexName
    } = request.params;
    try {
      const reindexService = (await getReindexService()).getScopedClient({
        savedObjects: getClient({
          includedHiddenTypes: [_upgradeAssistantPkgServer.REINDEX_OP_TYPE]
        }),
        dataClient: esClient,
        request
      });
      const body = await reindexService.getStatus(indexName);
      return response.ok({
        body
      });
    } catch (error) {
      if (error instanceof _elasticsearch.errors.ResponseError) {
        return (0, _server.handleEsError)({
          error,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(error);
    }
  });

  // Cancel reindex
  router.post({
    path: `${_common.REINDEX_SERVICE_BASE_PATH}/{indexName}/cancel`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es and saved object clients for authorization'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, async ({
    core
  }, request, response) => {
    const {
      savedObjects,
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      indexName
    } = request.params;
    const {
      getClient
    } = savedObjects;
    try {
      const reindexService = (await getReindexService()).getScopedClient({
        savedObjects: getClient({
          includedHiddenTypes: [_upgradeAssistantPkgServer.REINDEX_OP_TYPE]
        }),
        dataClient: esClient,
        request
      });
      await reindexService.cancel(indexName);
      return response.ok({
        body: {
          acknowledged: true
        }
      });
    } catch (error) {
      if (error instanceof _elasticsearch.errors.ResponseError) {
        return (0, _server.handleEsError)({
          error,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(error);
    }
  });
}